/* mdhutil.js
 * Cyberhole
 * Created 2025-07-11
 * Copyright © 2025 by Mark Damon Hughes. All Rights Reserved.
 * See BSDLicense.txt
 */

/* requires mdhutil */

/* eslint-env browser, es2020 */

"use strict";

//---------------------------------------
class AspicError extends Error {}

class AspicLoopException extends Error {}

const kAspic_Symbols = " ;\t\r\n()\""

const kAspicToken_ParenOpen = '(',
	kAspicToken_ParenClose = ')',
	kAspicToken_String = '"',
	kAspicToken_Number = '#',
	kAspicToken_Word = 'w';

class AspicToken {
	constructor(type, value=null, line=0) {
		this.type=type;
		this.value=value;
		this.line=line;
	}

	toString() {
		switch (this.type) {
			case kAspicToken_ParenOpen:
			case kAspicToken_ParenClose:
				return this.type;
			case kAspicToken_String:
				return '"'+this.value+'"'; // FIXME: escapes
			case kAspicToken_Number:
			case kAspicToken_Word:
				return this.value;
			default:
				throw new AspicError("Unknown token type "+this.type+" "+this.value+" at line "+this.line);
		}
	}

} // AspicToken

class AspicLambda {
	constructor(aspic, name, args, block) {
		this.aspic=aspic;
		this.name=name;
		this.args=args;
		this.block=block;
	}

	toString() {
		return "(define ("+this.name+" "+this.aspic.prettyPrint(this.args)+") "+
			this.aspic.prettyPrint(this.block)+")";
	}

} // AspicLambda

class Aspic {
	/** delegate receives all unknown function calls with (aspic,name,args) */
	constructor(delegate) {
		this.program=[];
		this.stack=[];
		this.stack.push({});
		this.functions={};
		this.line=0;
		this.delegate=delegate;
	}

	//---------------------------------------
	// Runtime functions

	assertList(arg) {
		if (isArray(arg)) {
			return arg;
		} else {
			this.error("Expected list, but got "+arg);
		}
	}

	assertName(arg) {
		if (arg instanceof AspicToken) {
			return arg;
		} else {
			this.error("Expected name, but got "+arg);
		}
	}

	assertNumber(arg) {
		if (isNumber(arg)) {
			return arg;
		} else {
			this.error("Expected number, but got "+arg);
		}
	}

	assertString(arg) {
		if (isString(arg)) {
			return arg;
		} else {
			this.error("Expected string, but got "+arg);
		}
	}

	error(msg) {
		const msgline="ERROR: "+msg+" at line "+this.line;
		this.log(msgline);
		throw new AspicError(msgline);
	}

	errorMethod(methodText, args) {
		this.error("Expected "+methodText+", but got "+args);
	}

	log(msg) {
		console.log(msg);
	}

	evalArg(arg) {
		//DLOG("evalArg "+arg);
		if (isNumber(arg) || isString(arg)) {
			return arg;
		} else if (isArray(arg)) {
			return this.evalStatement(arg);
		} else if (arg instanceof AspicToken) {
			return this.getVar(arg.value);
		} else {
			this.error("Don't know how to eval object "+arg);
		}
	}

	/** a list with a token first, is a statement.
	 * In any other case, it must be fully evaluated and returned.
	 */
	evalStatement(stmt) {
		//DLOG("eval "+stmt);
		if (stmt.length===0) {
			return stmt;
		}
		const first=stmt[0];
		let name=null;
		if (first instanceof AspicToken) {
			this.line=first.line;
			name=first.value;
		} else {
			// all constants?
			let con=true;
			for (const arg of stmt) {
				if (isArray(arg) || arg instanceof AspicToken) {
					con=false;
					break;
				}
			}
			if (con) {
				return stmt;
			}
			// eval everything, return as new list
			const out=[];
			for (const arg of stmt) {
				out.push(this.evalArg(arg));
			}
			return out;
		}

		// identify function
		const args=stmt.length===1 ? [] : stmt.slice(1);
		const func=ASPIC_BUILTINS[name];
		if (func) {
			return func(this, args);
		}
		const lam=this.functions[name];
		if (lam) {
			return this.evalLambda(lam, args);
		}
		if (this.delegate) {
			return this.delegate(this, name, args);
		}
		this.error("Unknown function "+name);
	}

	/** evaluates an AspicLambda, with given args. */
	evalLambda(lam, args) {
		//DLOG("evalLambda "+lam+" "+args);
		const ctx={};
		if (args.length!==lam.args.length) {
			this.error("Expected args "+lam.args+", but got "+args);
		}
		// assign all arguments into ctx
		for (let i=0; i<args.length; ++i) {
			let name=this.assertName(lam.args[i]);
			let ival=this.evalArg(args[i]);
			ctx[name]=ival;
		}
		// push local context, eval, pop out.
		this.stack.push(ctx);
		const rc=this.evalArg(lam.block);
		this.stack.pop();
		return rc;
	}

	getVar(name) {
		if (this.stack.length>1) {  // try local context
			const locals=this.stack[this.stack.length-1];
			const val=locals[name];
			if (val!==undefined) {
				return val;
			}
		}
		if (this.stack.length>0) {  // try global context
			const globals=this.stack[0];
			const val=globals[name];
			if (val!==undefined) {
				return val;
			}
		}
		this.error("Undefined variable "+name);
	}

	prettyPrint(arg) {
		if (isString(arg)) {
			return '"'+arg+'"'; // FIXME: escapes
		} else if (isNumber(arg)) {
			return arg.toString();
		} else if (arg instanceof AspicToken) {
			return arg.value;
		} else if (arg instanceof AspicLambda) {
			return arg.toString();
		} else if (isArray(arg)) {
			const buf=[];
			let first=true;
			buf.push("(");
			for (let obj of arg) {
				if (first) {
					first=false;
				} else {
					buf.push(" ");
				}
				buf.push(this.prettyPrint(obj));
			}
			buf.push(")");
			return buf.join("");
		} else {
			this.error("Don't know how to prettyPrint "+arg);
		}
	}

	run() {
		let rc=0;
		for (let stmt of this.program) {
			DLOG("RUN: "+stmt);
			if (isArray(stmt)) {
				rc=this.evalStatement(stmt);
			} else if (isString(stmt)) {
				// comment string
			} else {
				this.error("Expected statement, but got "+stmt);
			}
		}
		return rc;
	}

	setGlobalVar(name, val) {
		if (this.stack.length>=1) {
			this.stack[0][name]=val;
			//DLOG("global "+name+"="+val);
			return val;
		} else {
			this.error("No global variable context for "+name);
		}
	}

	setLocalVar(name, val) {
		if (this.stack.length>=2) {
			this.stack[ this.stack.length-1 ][name]=val;
			//DLOG("local "+name+"="+val);
			return val;
		} else {
			this.error("No local variable context for "+name);
		}
	}

	valueCompare(aval, bval) {
		if (isNumber(aval) && isNumber(bval)) {
			if (aval<bval) {
				return -1;
			} else if (aval>bval) {
				return 1;
			} else {
				return 0;
			}
		} else if (isString(aval) && isString(bval)) {
			return aval.localeCompare(bval);
		} else if (isArray(aval) && isArray(bval)) {
			const alen=aval.length, blen=bval.length, maxlen=Math.max(alen, blen);
			for (let i=0; i<maxlen; ++i) {
				if (i<alen && i<blen) {
					let cmp=this.valueCompare(aval[i], bval[i]);
					if (cmp!==0) {
						return cmp;
					}
				} else if (i>=alen) { // a was shorter
					return -1;
				} else if (i>=blen) { // a was shorter
					return 1;
				}
			}
			return 0;
		} else {
			this.error("Mismatched types for compare: "+this.prettyPrint(aval)+
				" with "+this.prettyPrint(bval));
		}
	}

	/** convenience to do eq, etc. */
	eqtest(name, args) {
		if (args.length!==2) {
			this.errorMethod("("+name+" A B)", args);
		}
		const a=this.evalArg(args[0]);
		const b=this.evalArg(args[1]);
		return this.valueCompare(a,b);
	}

	/** convenience to do items, first, last, etc.
	  * Indices are INCLUSIVE, 0-(vlen-1).
	  * negative index counts from the end.
	  **/
	sublist(name, args, i, j) {
		if (args.length!==1) {
			this.errorMethod("("+name+" VALUE)", args);
		}
		let val=this.evalArg(args[0]), vlen=val.length;
		while (i<0) {
			i+=vlen;
		}
		while (j<0) {
			j+=vlen;
		}
		if (i>=vlen || j>=vlen) {
			this.error("index out of range 0.."+vlen+": "+args);
		}
		DLOG("sublist "+this.prettyPrint(val)+", length "+vlen+", i="+i+", j="+j);
		if (isString(val)) {
			return val.substring(i,j+1);
		} else if (isArray(val)) {
			return val.slice(i,j+1);
		} else {
			this.error("Can only take items of string or list, not "+val);
		}
	}

	valueIsTrue(val) {
		if (isNumber(val)) {
			return val!==0;
		} else if (isString(val)) {
			return val.length!==0;
		} else if (isArray(val)) {
			return val.length!==0;
		} else {
			this.error("Don't know how to test truth of "+val);
		}
	}

	valueToNumber(val) {
		if (isNumber(val)) {
			return val;
		} else if (isString(val)) {
			const n=Number.parseFloat(val);
			return Number.isNaN(n) ? 0 : n;
		} else {
			this.error("Can only convert string to integer, not "+this.prettyPrint(val));
		}
	}

	valueToString(val) {
		if (isArray(val)) {
			return val.join(" ");
		} else {
			return val.toString();
		}
	}

	//---------------------------------------
	// Parser

	load(url, after) {
		ajax("GET", url, null, (result,error)=>{
			if (result) {
				this.parse(result);
				after();
			} else {
				this.error(error);
			}
		});
	}

	/** Reads all of `text` into program. */
	parse(text) {
		this._tokindex=0;
		this._tokens=[];
		this.line=0;
		this.tokenize(text);
		while (this._tokindex < this._tokens.length) {
			const tok=this._tokens[this._tokindex];
			++this._tokindex;
			if (tok.type===kAspicToken_ParenOpen) {
				this.program.push(this.parseList());
			} else if (tok.type===kAspicToken_String) {
				// use string as comment
				this.program.push(tok.value);
			} else {
				this.error("Expected list or string comment, but got token "+tok);
			}
		}
		this._tokindex=0;
		this._tokens=null;
		//DLOG("program="+this.prettyPrint(this.program));
	}

	parseList() {
		const list=[];
		while (this._tokindex < this._tokens.length) {
			const tok=this._tokens[this._tokindex];
			++this._tokindex;
			switch (tok.type) {
				case kAspicToken_ParenClose:
					return list;
				case kAspicToken_ParenOpen:
					list.push( this.parseList() );
					break;
				case kAspicToken_Number:
				case kAspicToken_String:
					list.push( tok.value );
					break;
				case kAspicToken_Word:
					list.push( tok );
					break;
				default:
					this.error("Unknown token "+tok+" in list "+list);
			}
		}
	}

	tokenize(text) {
		for (let ti=0; ti<text.length; ++ti) {
			let c=text.charAt(ti);
			//DLOG("tokenize line "+this.line+", "+ti+": "+c);
			switch (c) {
				case ';': {
					while (ti < text.length) {
						c=text.charAt(ti);
						if (c==='\r' || c==='\n') {
							++this.line;
							break;
						} else {
							++ti;
						}
					}
					//DLOG("    comment");
					break;
				}
				case ' ':
				case '\t':
					// whitespace, ignore
					//DLOG("    whitespace");
					break;
				case '\r':
				case '\n':
					++this.line;
					//DLOG("    newline");
					break;
				case '(': {
					const tok=new AspicToken(kAspicToken_ParenOpen, null, this.line);
					//DLOG("    OPEN "+tok);
					this._tokens.push(tok);
					break;
				}
				case ')': {
					const tok=new AspicToken(kAspicToken_ParenClose, null, this.line);
					//DLOG("    CLOSE "+tok);
					this._tokens.push(tok);
					break;
				}
				case '"': { // string
					++ti;
					const strchars=[];
					while (ti < text.length) {
						c=text.charAt(ti);
						if (c==='"') {
							break;
						} else if (c==='\\') {
							// string escapes
							if (ti+1 < text.length) {
								++ti;
								c=text.charAt(ti);
								switch (c) {
									case 't': strchars.push("\t"); break;
									case 'r': strchars.push("\r"); break;
									case 'n': strchars.push("\n"); break;
									// TODO: more escapes
									default: strchars.push(c); break;
								}
							} else { // trailing backslash
								strchars.push("\\");
							}
						} else { // not escape
							strchars.push(c);
						}
						++ti;
					}
					const tok=new AspicToken(kAspicToken_String, strchars.join(""), this.line);
					//DLOG("    STR "+strchars+"="+tok);
					this._tokens.push(tok);
					break;
				}
				case '-': case '.':
				case '0': case '1': case '2': case '3': case '4':
				case '5': case '6': case '7': case '8': case '9': { // number
					// TODO: scientific notation
					// number
					const numchars=[];
					numchars.push(c);
					while (ti+1 < text.length) {
						c=text.charAt(ti+1);
						if ((c>='0' && c<='9') || c==='.') {
							numchars.push(c);
							++ti;
						} else {
							break;
						}
					}
					const numstr=numchars.join("");
					if (numstr==="-") {
						const tok=new AspicToken(kAspicToken_Word, numstr, this.line);
						//DLOG("    WORD "+tok);
						this._tokens.push(tok);
					} else {
						let n=Number.parseFloat(numstr);
						if ( ! Number.isNaN(n)) {
							const tok=new AspicToken(kAspicToken_Number, n, this.line);
							//DLOG("    NUM "+numchars+"="+tok);
							this._tokens.push(tok);
						} else {
							this.error("Invalid number "+numstr);
						}
					}
					break;
				}
				default: { // word
					const wordchars=[];
					wordchars.push(c);
					while (ti+1 < text.length) {
						c=text.charAt(ti+1);
						if (kAspic_Symbols.indexOf(c)<0) {
							wordchars.push(c);
							++ti;
						} else {
							break;
						}
					}
					const tok=new AspicToken(kAspicToken_Word, wordchars.join(""), this.line);
					//DLOG("    WORD "+wordchars+"="+tok);
					this._tokens.push(tok);
				}
			}
		}
		//DLOG("tokens="+this._tokens.join("\n"));
	}

} // Aspic

const ASPIC_BUILTINS={

	add: (aspic, args)=>{
		if (args.length<2) {
			aspic.errorMethod("(add X Y...)", args);
		}
		let total=0;
		for (let y of args) {
			total+=aspic.valueToNumber(aspic.evalArg(y));
		}
		return total;
	},

	sub: (aspic, args)=>{
		if (args.length<2) {
			aspic.errorMethod("(sub X Y...)", args);
		}
		let total=aspic.valueToNumber(aspic.evalArg(args[0]));
		for (let i=1; i<args.length; ++i) {
			total-=aspic.valueToNumber(aspic.evalArg(args[i]));
		}
		return total;
	},

	mul: (aspic, args)=>{
		if (args.length<2) {
			aspic.errorMethod("(mul X Y...)", args);
		}
		let total=aspic.valueToNumber(aspic.evalArg(args[0]));
		for (let i=1; i<args.length; ++i) {
			total*=aspic.valueToNumber(aspic.evalArg(args[i]));
		}
		return total;
	},

	div: (aspic, args)=>{
		if (args.length<2) {
			aspic.errorMethod("(div X Y...)", args);
		}
		let total=aspic.valueToNumber(aspic.evalArg(args[0]));
		for (let i=1; i<args.length; ++i) {
			const y=aspic.valueToNumber(aspic.evalArg(args[i]));
			if (y!==0) {
				total/=y;
			} else {
				aspic.error("Divide by 0");
			}
		}
		return total;
	},

	mod: (aspic, args)=>{
		if (args.length<2) {
			aspic.errorMethod("(mod X Y...)", args);
		}
		let total=aspic.valueToNumber(aspic.evalArg(args[0]));
		for (let i=1; i<args.length; ++i) {
			const y=aspic.valueToNumber(aspic.evalArg(args[i]));
			if (y!==0) {
				total%=y;
			} else {
				aspic.error("Modulo by 0");
			}
		}
		return total;
	},

	and: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(and A...)", args);
		}
		let rc=0;
		for (let arg of args) {
			rc=aspic.evalArg(arg);
			if ( ! aspic.valueIsTrue(rc)) {
				return 0;
			}
		}
		return rc;
	},

	not: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(not X)", args);
		}
		return aspic.valueIsTrue(aspic.evalArg(arg)) ? 0 : 1;
	},

	or: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(or A...)", args);
		}
		let rc=0;
		for (let arg of args) {
			rc=aspic.evalArg(arg);
			if (aspic.valueIsTrue(rc)) {
				return rc;
			}
		}
		return rc;
	},

	apply: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(apply FNAME A...)", args);
		}
		let fname=aspic.evalArg(args[0]);
		if (isString(fname)) {
			const stmt=[];
			stmt.push(new AspicToken(kAspicToken_Word, fname, aspic.line));
			for (let i=1; i<args.length; ++i) {
				stmt.push(args[i]); // not eval
			}
			return aspic.evalStatement(stmt);
		} else {
			aspic.error("Can only apply string function name, not "+args);
		}
	},

	begin: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(begin VALUE...)", args);
		}
		let rc=0;
		for (let arg of args) {
			rc=aspic.evalArg(arg);
		}
		return rc;
	},

	'break': (aspic, args)=>{
		if (args.length!==0) {
			aspic.errorMethod("(break)", args);
		}
		throw new AspicLoopException("break");
	},

	chr: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(chr VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		if (isNumber(val)) {
			return String.fromCodePoint(Math.floor(val));
		} else {
			aspic.error("Can only take chr of number, not "+val);
		}
	},

	ord: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(ord VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		if (isString(val) && val.length>=1) {
			return val.codePointAt(0);
		} else {
			aspic.error("Can only take ord of string length 1+, not "+val);
		}
	},

	cond: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(cond (TEST THEN)...)", args);
		}
		let rc=0;
		for (let pair of args) {
			if (isArray(pair) && pair.length===2) {
				if (aspic.valueIsTrue(aspic.evalArg(pair[0]))) {
					rc=aspic.evalArg(pair[1]);
					break;
				}
			} else {
				aspic.error("Expected (TEST THEN), but got "+aspic.prettyPrint(pair));
			}
		}
		return rc;
	},

	define: (aspic, args)=>{
		if (args.length!==2) {
			aspic.errorMethod("Block: (define (NAME ARGS) BLOCK)", args);
		}
		const params=aspic.assertList(args[0]);
		if (params.length===0) {
			aspic.errorMethod("Params: (define (NAME ARGS) BLOCK)", params);
		}
		const name=aspic.assertName(params[0]);
		if (ASPIC_BUILTINS[name]) {
			aspic.error("Attempt to redefine builtin "+name);
		}
		if (aspic.functions[name]) {
			aspic.error("Attempt to redefine function "+name);
		}
		const lam=new AspicLambda(aspic, name, aspic.assertList(params.slice(1)),
			aspic.assertList(args[1]));
		aspic.functions[name]=lam;
		DLOG("DEFINE "+name+"="+aspic.prettyPrint(lam));
		return 0;
	},

	display: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(display VALUE...)", args);
		}
		const msg=[];
		for (let arg of args) {
			const value=aspic.evalArg(arg);
			msg.push(value);
		}
		const line=msg.join("");
		aspic.log(line);
		if (terminal) {
			terminal.print(line+"\n");
		}
		return 0;
	},

	error: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(display VALUE...)", args);
		}
		const msg=[];
		for (let arg of args) {
			const value=aspic.evalArg(arg);
			msg.push(value);
		}
		aspic.error(msg.join(""));
		return 0;
	},

	first: (aspic, args)=>{
		let rc=aspic.sublist("first", args, 0, 0);
		if (rc.length===1) {
			return rc[0];
		} else {
			aspic.error("Expected first 1 element, but got "+aspic.prettyPrint(rc));
		}
	},

	butfirst: (aspic, args)=>{
		return aspic.sublist("butfirst", args, 1, -1);
	},

	last: (aspic, args)=>{
		let rc=aspic.sublist("last", args, -1, -1);
		if (rc.length===1) {
			return rc[0];
		} else {
			aspic.error("Expected last 1 element, but got "+aspic.prettyPrint(rc));
		}
	},

	butlast: (aspic, args)=>{
		return aspic.sublist("butlast", args, 0, -2);
	},

	items: (aspic, args)=>{
		if (args.length!==3) {
			aspic.errorMethod("(items A FROM TO)", args);
		}
		let i=aspic.evalArg(args[1]), j=aspic.evalArg(args[2]);
		return aspic.sublist("items", args.slice(0,1), i, j);
	},

	eq: (aspic, args)=>{ return aspic.eqtest("eq", args)===0 ? 1 : 0; },
	ne: (aspic, args)=>{ return aspic.eqtest("ne", args)!==0 ? 1 : 0; },
	gt: (aspic, args)=>{ return aspic.eqtest("gt", args)>0 ? 1 : 0; },
	ge: (aspic, args)=>{ return aspic.eqtest("ge", args)>=0 ? 1 : 0; },
	lt: (aspic, args)=>{ return aspic.eqtest("lt", args)<0 ? 1 : 0; },
	le: (aspic, args)=>{ return aspic.eqtest("le", args)<=0 ? 1 : 0; },

	global: (aspic, args)=>{
		if (args.length!==2) {
			aspic.errorMethod("(global NAME VALUE)", args);
		}
		const name=aspic.assertName(args[0]);
		const value=aspic.evalArg(args[1]);
		return aspic.setGlobalVar(name, value);
	},

	'if': (aspic, args)=>{
		if (args.length!==3) {
			aspic.errorMethod("(if TEST THEN ELSE)", args);
		}
		let test=aspic.valueIsTrue(aspic.evalArg(args[0]));
		if (test) {
			return aspic.evalArg(args[1]);
		} else {
			return aspic.evalArg(args[2]);
		}
	},

	'int': (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(int VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		return Math.floor(aspic.valueToNumber(val));
	},

	indexof: (aspic, args)=>{
		if (args.length!==2) {
			aspic.errorMethod("(indexof NEEDLE HAYSTACK)", args);
		}
		const needle=aspic.evalArg(args[0]);
		const haystack=aspic.evalArg(args[1]);
		if (isArray(haystack) || (isString(haystack) && isString(needle)) ) {
			return haystack.indexOf(needle);
		} else {
			aspic.error("Can only indexof any in list or string in string, not "+args);
		}
	},

	islist: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(islist VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		return isArray(val) ? 1 : 0;
	},

	isnumber: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(isnumber VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		return isNumber(val) ? 1 : 0;
	},

	isstring: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(isstring VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		return isString(val) ? 1 : 0;
	},

	length: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(length VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		if (isString(val) || isArray(val)) {
			return val.length;
		} else {
			aspic.error("Can only take length of string or list, not "+val);
		}
	},

	list: (aspic, args)=>{
		const ls=arrayDim(args.length, 0);
		for (let i=0; i<args.length; ++i) {
			ls[i]=aspic.evalArg(args[i]);
		}
		return ls;
	},

	listref: (aspic, args)=>{
		if (args.length!==2) {
			aspic.errorMethod("(listref LS INDEX)", args);
		}
		const ls=aspic.evalArg(args[0]);
		let index=aspic.evalArg(args[1]);
		if (isArray(ls) && isNumber(index)) {
			index=Math.floor(index);
			if (index>=0 && index<ls.length) {
				return ls[index];
			} else {
				aspic.error("Index "+index+" out of range 0.."+ls.length);
			}
		} else {
			aspic.error("Can only listref array with number index, not "+args);
		}
	},

	listset: (aspic, args)=>{
		if (args.length!==3) {
			aspic.errorMethod("(listsset LS INDEX VALUE)", args);
		}
		const ls=aspic.evalArg(args[0]);
		let index=aspic.evalArg(args[1]);
		const value=aspic.evalArg(args[2]);
		if (isArray(ls) && isNumber(index)) {
			index=Math.floor(index);
			if (index>=0 && index<ls.length) {
				ls[index]=value;
				return value;
			} else {
				aspic.error("Index "+index+" out of range 0.."+ls.length);
			}
		} else {
			aspic.error("Can only listset array with number index, not "+args);
		}
	},

	lower: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(lower VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		if (isString(val)) {
			return val.toLocaleLowerCase();
		} else {
			aspic.error("Can only take lower of string, not "+val);
		}
	},

	// title-case changes every char, if preceded by a letter, becomes lower-case;
	// otherwise it is upper-cased.
	title: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(title VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		if (isString(val)) {
			if (val.length===0) {
				return "";
			} else {
				let out=arrayDim(val.length,""), lastLetter=false;
				for (let i=0; i<val.length; ++i) {
					const c=val.charAt(i);
					if (lastLetter) {
						out[i]=c.toLocaleLowerCase();
					} else {
						out[i]=c.toLocaleUpperCase();
					}
					lastLetter=isAlpha(c);
				}
				return out.join("");
			}
		} else {
			aspic.error("Can only take title of string, not "+val);
		}
	},

	upper: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(upper VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		if (isString(val)) {
			return val.toLocaleUpperCase();
		} else {
			aspic.error("Can only take upper of string, not "+val);
		}
	},

	makelist: (aspic, args)=>{
		if (args.length!==2) {
			aspic.errorMethod("(makelist LEN FILL)", args);
		}
		const len=aspic.evalArg(args[0]);
		const fill=aspic.evalArg(args[1]);
		if ((isNumber(len) && len>=1) && (isNumber(fill) || isString(fill))) {
			return arrayDim(len, fill);
		} else {
			aspic.error("Can only makelist of positive number length, string or number fill, not "+args);
		}
	},

	merge: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(merge VALUE...)", args);
		}
		const ls=[];
		for (let arg of args) {
			const value=aspic.evalArg(arg);
			if (isArray(value)) {
				for (let sub of value) {
					ls.push(sub);
				}
			} else {
				ls.push(value);
			}
		}
		return ls;
	},

	reverse: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(reverse VALUE)", args);
		}
		const value=aspic.evalArg(args[0]);
		if (isArray(value)) {
			return value.reverse();
		} else if (isString(value)) {
			const vlen=value.length, out=arrayDim(vlen, "");
			for (let i=0; i<vlen; ++i) {
				out[vlen-i]=value[i];
			}
			return out.join("");
		} else {
			aspic.error("Can only reverse string or list, not "+args);
		}
	},

	num: (aspic, args)=>{
		if (args.length!==1) {
			aspic.errorMethod("(num VALUE)", args);
		}
		const val=aspic.evalArg(args[0]);
		return aspic.valueToNumber(val);
	},

	set: (aspic, args)=>{
		if (args.length!==2) {
			aspic.errorMethod("(set NAME VALUE)", args);
		}
		const name=aspic.assertName(args[0]);
		const value=aspic.evalArg(args[1]);
		return aspic.setLocalVar(name, value);
	},

	str: (aspic, args)=>{
		if (args.length===0) {
			aspic.errorMethod("(str VALUE...)", args);
		} else if (args.length===1) {
			const val=aspic.evalArg(args[0]);
			return aspic.valueToString(val);
		} else {
			const buf=[];
			for (let arg of args) {
				const val=aspic.evalArg(arg);
				buf.push(aspic.valueToString(val));
			}
			return buf.join("");
		}
	},

	'while': (aspic, args)=>{
		if (args.length!==2) {
			aspic.errorMethod("(while TEST LOOP)", args);
		}
		let rc=0;
		try {
			while (aspic.valueIsTrue( aspic.evalArg(args[0]))) {
				rc=aspic.evalArg(args[1]);
			}
		} catch (e) {
			if (e instanceof AspicLoopException) {
				// terminated by break
			} else {
				throw e;
			}
		}
		return rc;
	},

}; // ASPIC_BUILTINS
// aliases
(function() {
	const ASPIC_ALIASES = {
		'car':	'first',
		'cdr':	'butfirst',
		'+':	'add',
		'-':	'sub',
		'*':	'mul',
		'/':	'div',
		'%':	'mod',
		'==':	'eq',
		'!=':	'ne',
		'<':	'lt',
		'<=':	'le',
		'>':	'gt',
		'>=':	'ge',
	};
	for (let key in ASPIC_ALIASES) {
		ASPIC_BUILTINS[key] = ASPIC_BUILTINS[ ASPIC_ALIASES[key] ];
	}
})();

