/* gridmap.js
 * Cyberhole
 * Created 2020-08-15 10:31:11
 * Updated 2025-08-17
 * Copyright © 2020,2025 by Mark Damon Hughes. All Rights Reserved.
 * See BSDLicense.txt
 */

// requires mdhutil.js

/* eslint-env browser, es2020 */

"use strict";

//---------------------------------------
// MathUtil

const Dir={
	N: 0,
	E: 1,
	S: 2,
	W: 3,
	U: 4,
	D: 5,
	count: 4,
	opposite: [2, 3, 0, 1, 5, 4],
	delta: [ [0, -1], [1, 0], [0, 1], [-1, 0], [0,0], [0,0] ],
	names: ["North", "East", "South", "West", "Up", "Down", ],
	arrows: [
		"\u2191",	// 0 = ↑
		"\u2192",	// 1 = →
		"\u2193",	// 2 = ↓
		"\u2190",	// 3 = ←
		"<",
		">",
	],
	forDelta: (delta)=>{
		if (delta[0]!==0 && Math.abs(delta[0])>Math.abs(delta[1])) {
			return (delta[0]>0) ? Dir.E : Dir.W;
		} else {
			return (delta[1]>0) ? Dir.S : Dir.N;
			// returns N for 0,0.
		}
	},
};

function pointAdd(a, b) {
	return [a[0]+b[0], a[1]+b[1]];
}

/** Taxicab distance */
function pointDist(a, b) {
	const xterm=a[0]-b[0], yterm=a[1]-b[1];
	return Math.abs(xterm)+Math.abs(yterm);
}

function pointEquals(a, b) {
	return a && b && a[0]===b[0] && a[1]===b[1];
}

function pointMult(a, n) {
	return [a[0]*n, a[1]*n];
}

function pointSubtract(a, b) {
	return [a[0]-b[0], a[1]-b[1]];
}

/** Returns a list of points from `pt0` to `pt1`, with "fat" diagonals. */
function pointsOnLine(pt0, pt1) {
	const points=[pt0];
	if (pointEquals(pt0, pt1)) {
		return points;
	}
	let delta=[pt1[0]-pt0[0], pt1[1]-pt0[1]];
	const divisor=Math.max(Math.abs(delta[0]), Math.abs(delta[1]))*2.0;
	delta=[delta[0]/divisor, delta[1]/divisor];
	let pt=[pt0[0]+0.5, pt0[1]+0.5];
	let lastPt=pt0;
	do {
		pt=pointAdd(pt, delta);
		const ipt=[Math.floor(pt[0]), Math.floor(pt[1])];
		if ( ! pointEquals(ipt, points[points.length-1])) {
			points.push(ipt);
		}
		// fill in diagonals
		// FIXME: was correct? if (Math.abs(ipt[0]-lastPt[0]) + Math.abs(ipt[1]-lastPt[1]) >1) {
		if ((Math.abs(ipt[0]-lastPt[0]) + Math.abs(ipt[1]-lastPt[1])) >1) {
			if (Math.abs(delta[0])>Math.abs(delta[1])) {
				const yoff=Math.sign(delta[1]);
				const offPt=[lastPt[0], lastPt[1]+yoff];
				if ( ! pointEquals(offPt, points[points.length-1])) {
					points.push(offPt);
				}
			} else {
				const xoff=Math.sign(delta[0]);
				const offPt=[lastPt[0]+xoff, lastPt[1]];
				if ( ! pointEquals(offPt, points[points.length-1])) {
					points.push(offPt);
				}
			}
		}
		lastPt=ipt;
	} while ( ! pointEquals(lastPt, pt1));
	return points;
}

/** Returns true if `outer` contains `pt`. */
function rectContainsPoint(outer, pt) {
	return pt[0] >= outer[0] && pt[1] >= outer[1] && pt[0] < outer[0]+outer[2] && pt[1] < outer[1]+outer[3];
}

/** Returns a rectangle inset from `rect` by `n` units from all sides. */
function rectInset(rect, n) {
	return [rect[0]+n, rect[1]+n, rect[2]-n*2, rect[3]-n*2];
}

/** Returns true if rects `a` and `b` have any intersection. */
function rectIntersectsRect(a, b) {
	// const aleft=a[0], atop=a[1], aright=a[0]+a[2], abottom=a[1]+a[3];
	// const bleft=b[0], btop=b[1], bright=b[0]+b[2], bbottom=b[1]+b[3];
	//return aleft < bright && aright > bleft && atop < bbottom && abottom > btop;
	return a[0] < b[0]+b[2] && a[0]+a[2] > b[0] && a[1] < b[1]+b[3] && a[1]+a[3] > b[1];
}

function rectOrigin(rect) {
	return [rect[0], rect[1]];
}

function rectSize(rect) {
	return [rect[2], rect[3]];
}

/** Returns an array of deltas to move from a to b, including diagonals.
* If `tight` is true, only exact directions are allowed.
*/
function deltasTowards(a, b) {
	const deltas=[];
	const dist=pointDist(a, b);
	if (b[0] < a[0]) {
		if (dist <= 1) {
			deltas.push( [-1,0] );
		} else {
			deltas.push( [-1,0], [-1,1], [-1,1] );
		}
	} else if (b[0] > a[0]) {
		if (dist <= 1) {
			deltas.push( [1,0] );
		} else {
			deltas.push( [1,0], [1,1], [1,-1] );
		}
	}
	if (b[1] < a[1]) {
		if (dist <= 1) {
			deltas.push( [0,-1] );
		} else {
			deltas.push( [0,-1], [1,-1], [-1,-1] );
		}
	} else if (b[1] > a[1]) {
		if (dist <= 1) {
			deltas.push( [0,1] );
		} else {
			deltas.push( [0,1], [1,1], [-1,1] );
		}
	}
	return deltas;
}

//---------------------------------------
// GridMap

class GridMapError extends Error {}

const kHookForDir=["hook_n", "hook_e", "hook_s", "hook_w"];

/** Bitmask of flag integers */
const kGridFlag={
	visible: 0x01,
	opaque: 0x02,
	seen: 0x04,
	walkable: 0x08,
};

/** Rectangular container.
 * Size is [width,height].
 * Each grid is an object with arbitary keys, typically {t=terrain, flag=bitmask}.
 */
class GridMap {
	constructor(size, fill) {
		this.size=size;
		this.density=1.0;
		const grids=[];
		for (let y=0; y<size[1]; ++y) {
			for (let x=0; x<size[0]; ++x) {
				grids.push( Object.assign({}, fill) );
			}
		}
		this.grids=grids;
	}

	bounds() {
		return [0, 0, this.size[0], this.size[1]];
	}

	gridAt(pt) {
		if (this.inBounds(pt)) {
			const i=pt[0] + pt[1]*this.size[0];
			return this.grids[i];
		} else {
			return null;
		}
	}

	setGridAt(pt, g) {
		if (this.inBounds(pt)) {
			const i=pt[0]+ pt[1]*this.size[0];
			this.grids[i]=g;
		} else {
			return null;
		}
	}

	inBounds(pt) {
		return pt[0] >= 0 && pt[0] < this.size[0] && pt[1] >= 0 && pt[1] < this.size[1];
	}

	/** Copies elements from fill into each grid in the border of rect. */
	drawRect(rect, fill) {
		this.filter(rect, (map, pt, g)=>{
			if (pt[0] === rect[0] || pt[0] === rect[0]+rect[2]-1 || pt[1] === rect[1] || pt[1] === rect[1]+rect[3]-1) {
				this.fillGridAt(pt, fill);
			}
		});
	}

	/** Copies elements from fill into each grid in rect. */
	fillRect(rect, fill) {
		this.filter(rect, (map, pt, g)=>{
			this.fillGridAt(pt, fill);
		});
	}

	/** Copies elements from fill into grid at pt. */
	fillGridAt(pt, fill) {
		const g=this.gridAt(pt);
		if (g) {
			Object.assign(g, fill);
		}
	}

	/** Calls func(map, pt, g) on every grid in rect. */
	filter(rect, func) {
		for (let y=rect[1]; y<rect[1]+rect[3]; ++y) {
			for (let x=rect[0]; x<rect[0]+rect[2]; ++x) {
				const pt=[x,y];
				const g=this.gridAt(pt);
				if (g) {
					func(this, pt, g);
				}
			}
		}
	}

	/** Returns a list of points for which `func(map2, pt2, g2)` returns true. */
	findGrids(func) {
		const points=[];
		this.filter(this.bounds(), (map, pt, g)=>{
			if (func(map, pt, g)) {
				points.push(pt);
			}
		});
		return points;
	}

	/** Returns a list of points which have the same values as match. */
	findGridsMatch(match) {
		const points=[];
		this.filter(this.bounds(), (map, pt, g)=>{
			let ok=true;
			for (const key in match) {
				if (g[key] !== match[key]) {
					ok=false;
				}
			}
			if (ok) {
				points.push(pt);
			}
		});
		if (points.length === 0) {
			return null;
		}
		return points;
	}

	gridFlagTest(g, f) {
		return (g.flag & f) !== 0;
	}

	gridFlagReset(g, f) {
		g.flag &= ~f;
	}

	gridFlagSet(g, f) {
		g.flag |= f;
	}

	toString(func) {
		let s="";
		this.filter(this.bounds(), (map, pt, g)=>{
			s+=func ? func(g) : g.t;
			if (pt[0]===this.size[0]-1) {
				s+="\n";
			}
		});
		return s;
	}

	/** Visibility calculations.
	* loc: Visibility center.
	* viewDist: Maximum view distance to compute.
	* terIsOpaque: Function to take g and return true if opaque. Example:
	* function terIsOpaque(g) {
	* 	const tdata=kTileData[g.t], fdata=g.f ? kTileData[g.f] : null;
	* 	if (tdata === undefined || fdata === undefined) {
	* 		throw new GridMapError("Invalid grid", g);
	* 		return false;
	* 	}
	* 	return tdata.opaque || (fdata && fdata.opaque);
	* }
	*/
	// TODO: terIsOpaque should take dx,dy for node visibility
	visicalcFrom(loc, viewDist, terIsOpaque) {
		// clear visibility
		for (let y=0; y<this.size[1]; ++y) {
			for (let x=0; x<this.size[0]; ++x) {
				const g=this.gridAt([x,y]);
				this.gridFlagReset(g, kGridFlag.opaque);
				this.gridFlagReset(g, kGridFlag.visible);
			}
		}

		// your position
		const g=this.gridAt(loc);
		this.gridFlagReset(g, kGridFlag.opaque);
		this.gridFlagSet(g, kGridFlag.visible | kGridFlag.seen);

		for (let dist=1; dist <= viewDist; ++dist) {
			// draw top & bottom
			for (let dx=-dist, x=loc[0]+dx; dx<=dist; ++dx, ++x) {
				for (let dy=-dist, y=loc[1]+dy; dy<=dist; dy+=dist+dist, y=loc[1]+dy) {
					this._visicalcFromDeltaTo(loc, [dx, dy], [x, y], viewDist, terIsOpaque);
				}
			}
			// draw sides
			for (let dy=-dist+1, y=loc[1] + dy; dy<dist; ++dy, ++y) {
				for (let dx=-dist, x=loc[0] + dx; dx<=dist; dx += dist+dist, x=loc[0]+dx) {
					this._visicalcFromDeltaTo(loc, [dx, dy], [x, y], viewDist, terIsOpaque);
				}
			}
		}
	}

	/** Visibility calculations.
	* loc = normalized coord of the player or visibility epicenter
	* delta = delta from loc to examine
	* pt = point to examine
	*/
	_visicalcFromDeltaTo(loc, delta, pt, viewDist, terIsOpaque) {
		const g=this.gridAt(pt);
		if (!g) {
			return;
		}

		// Calculate the two points on the sight path to check
		const adx=Math.abs(delta[0]), ady=Math.abs(delta[1]);
		const sdx=Math.sign(delta[0]), sdy=Math.sign(delta[1]);

		// fake-hypotenuse view distance
		if (Math.min(adx + ady/2, adx/2 + ady) > viewDist) {
			return;
		}

		const x1=-sdx+delta[0]+loc[0];
		const y1=-sdy+delta[1]+loc[1];
		let x2, y2;

		if (ady>adx) {
			x2=delta[0]+loc[0];
			y2= -sdy+delta[1]+loc[1];
		} else if (ady<adx) {
			x2= -sdx+delta[0]+loc[0];
			y2=delta[1]+loc[1];
		} else {
			x2=x1;
			y2=y1;
		}
		const g1=this.gridAt([x1, y1]);
		const g2=this.gridAt([x2, y2]);

		// are both sides blocked?
		if ( (!g1 || this.gridFlagTest(g1, kGridFlag.opaque) ) && (!g2 || this.gridFlagTest(g2, kGridFlag.opaque) ) ) {
			this.gridFlagSet(g, kGridFlag.opaque);
			this.gridFlagReset(g, kGridFlag.visible);
			return;
		}

		this.gridFlagSet(g, kGridFlag.visible | kGridFlag.seen);

		// set opacity of grid based on tile & feature
		if (terIsOpaque(g)) {
			this.gridFlagSet(g, kGridFlag.opaque);
		} else {
			this.gridFlagReset(g, kGridFlag.opaque);
		}
	}

} // class GridMap

//_______________________________________
/** NodeGridMap adds node-based generators. */
class NodeGridMap extends GridMap {
	constructor(size) {
		super(size, objectMake("t"," ", Dir.N,0, Dir.E,0, Dir.S,0, Dir.W,0));
	}

	/** Links a node-based grid g[Dir.*] to its neighbor, and opposite back. */
	nodeLink(pt, d, value) {
		const g=this.gridAt(pt);
		if (g) {
			g[d]=value;
		}
		const g2=this.gridAt(pointAdd(pt, Dir.delta[d]));
		if (g2) {
			g2[ Dir.opposite[d] ]=value;
		}
	}

	// TODO: cellsize
	nodeToString() {
		let s="";
		for (let y=0; y<this.size[1]; ++y) {
			// top row
			for (let x=0; x<this.size[0]; ++x) {
				const g=this.gridAt([x,y]);
				const gn=this.gridAt([x,y-1]);
				const gw=this.gridAt([x-1,y]);
				// bitmask 1=N, 2=E, 4=S, 8=W
				// crossbar:
				//     gn.w
				// gw.n  +  g.n
				//      g.w
				s += kRunesNum.box[ ((gn && gn[Dir.W]) || !gn ? 0 : 1)+
						(g[Dir.N] ? 0 : 2)+
						(g[Dir.W] ? 0 : 4)+
						((gw && gw[Dir.N]) || !gw ? 0 : 8) ];
				// north
				s += g[Dir.N] ? "  " : kRunesNum.box[10]+kRunesNum.box[10];
				if (x === this.size[0]-1) {
					s += kRunesNum.box[ (y > 0 ? 1 : 0)+
						4+ (g[Dir.N] ? 0 : 8) ] + "\n";
				}
			}

			// middle row
			for (let x=0; x<this.size[0]; ++x) {
				const g=this.gridAt([x,y]);
				const i=g[Dir.N]*1 + g[Dir.E]*2 + g[Dir.S]*4 + g[Dir.W]*8;
				s+=(g[Dir.W] ? " " : kRunesNum.box[5]) + g.t+" "; // + i.toString(16);
			}
			s+=kRunesNum.box[5]+"\n";
		}
		for (let x=0, y=this.size[1]-1; x<this.size[0]; ++x) {
			const g=this.gridAt([x,y]);
			s+=kRunesNum.box[ (g[Dir.W] ? 0 : 1)+
				2+
				(x > 0 ? 8 : 0) ] + kRunesNum.box[10]+kRunesNum.box[10];
		}
		s+=kRunesNum.box[9]+"\n";
		return s;
	}

}

//_______________________________________
/** TileGridMap adds tile-based generators. */
class TileGridMap extends GridMap {
	constructor(size, fill) {
		super(size, fill);
	}

	/** Returns a histogram of the most frequent terrains in 1 square radius. */
	histogram(pt) {
		const hist={};
		for (let dy= -1; dy<=1; ++dy) {
			for (let dx= -1; dx<=1; ++dx) {
				const pt2=pointAdd(pt, [dx,dy]);
				const g2=this.gridAt(pt2);
				if (g2) {
					hist[g2.t]=(hist[g2.t] || 0)+1;
				}
			}
		}
		return hist;
	}

	/** For each grid in rect, replaces g.t with random, weighted selection of g.t from radius 2. */
	weightedAverage(rect) {
		this.filter(rect, (map, pt, g)=>{
			const bucket=[];
			for (let dy= -2; dy<=2; ++dy) {
				for (let dx= -2; dx<=2; ++dx) {
					const weight=3 - (Math.abs(dx) + Math.abs(dy));
					const pt2=pointAdd(pt, [dx,dy]);
					const g2=this.gridAt(pt2);
					if (g2 && weight > 0) {
						for (let i=0; i<weight; ++i) {
							bucket.push(g2.t);
						}
					}
				}
			}
			g.t=rndArrayChoose(bucket);
		});
	}

	/** Fills the map with a maze, setting g[0] to given floor (bgter) & wall (fgter) values */
	genMaze() {
		this.fillRect(this.bounds(), { t: this.fgter });
		// sizes must always be odd
		const width=this.size[0] + (this.size[0]%2===0?-1:0), height=this.size[1] + (this.size[1]%2===0?-1:0);
		this.startPt=[rnd(Math.floor(width/2)-2)*2+1, 1];
		this.gridAt(this.startPt).t=this.bgter;

		const endRow=height-2;
		this.endPt=null;
		let pt=this.startPt;
		//DLOG("initMaze start at "+pt);
		// connect all rooms (odd-numbered indices), through tunnels (even-numbered indices)
		for (;;) {
			//DLOG("map:\n"+this.toString((g)=>{ return g.t > 32 ? String.fromCharCode(g.t) : String.fromCharCode(64+g.t); } ));
			const g=this.gridAt(pt);
			const dirs=this.mazeUnconnectedDirs(pt, width, height);
			if (dirs.length > 0) {
				const d=rndArrayChoose(dirs);
				// corridor
				const pt1=pointAdd(pt, Dir.delta[d]);
				this.fillGridAt(pt1, { t: this.bgter });
				const pt2=pointAdd(pt1, Dir.delta[d]);
				this.fillGridAt(pt2, { t: this.bgter });
				pt=pt2;
				//DLOG("move "+pt+" by "+d+" to "+pt2);
				if (pt[1] === endRow) {
					this.endPt=pt;
				}
			} else { // pick a new connected room with unconnected neighbors and try again
				const connected=[];
				for (let y=1; y<height-1; y+=2) {
					for (let x=1; x<width-1; x+=2) {
						const pt2=[x,y];
						const g2=this.gridAt(pt2);
						if (g2 && g2.t === this.bgter && this.mazeUnconnectedDirs(pt2, width, height).length > 0) {
							connected.push(pt2);
						}
					}
				}
				if (connected.length > 0) {
					pt=rndArrayChoose(connected);
					//DLOG("teleport to "+pt);
				} else {
					break;
				}
			}
		}
		if ( ! this.endPt) {
			throw new GridMapError("Warning: Map never wrote an end point");
			this.endPt=[rnd(Math.floor(width/2)-2)*2+1, endRow];
		}
	}

	/* Returns a list of dirs which have unconnected neighbor cells. */
	mazeUnconnectedDirs(pt, width, height) {
		const unconnected=[];
		for (let d=0; d < Dir.count; ++d) {
			const pt1=pointAdd(pt, Dir.delta[d]);
			const g1=this.gridAt(pt1);
			const pt2=pointAdd(pt1, Dir.delta[d]);
			if (rectContainsPoint([0,0,width,height], pt2)) {
				const g2=this.gridAt(pt2);
				if (g1 && g1.t !== this.bgter && g2 && g2.t !== this.bgter) {
					unconnected.push(d);
				}
			}
		}
		//DLOG("found "+unconnected+" around "+pt);
		return unconnected;
	}

	genWorld() {
		// round size down to even power of 2
		const evensize=roundPowerOf2(Math.max(this.size[0], this.size[1]));
		this.fillRect(this.bounds(), { t: this.bgter, f: null });
		const step=evensize >> 1;
		this.startPt=[step, step];
		this.fillGridAt(this.startPt, { t: this.fgter });
		this.worldFractal(evensize, step>>1);
	}

	worldFractal(evensize, step) {
		for (let y=0; y < evensize; y += step) {
			for (let x=0; x < evensize; x += step) {
				// add random offsets
				let cx=x, cy=y;
				if (rnd(2) === 0) {
					cx += step;
				}
				if (rnd(2) === 0) {
					cy += step;
				}
				// Truncate to nearest multiple of step*2,
				// since step*2 is the previous detail level calculated.
				cx=Math.floor(cx / (step+step)) * (step+step);
				cy=Math.floor(cy / (step+step)) * (step+step);
				// Read from the randomized cell
				// Assume the world beyond the boundaries is nothing but endless
				// water.
				let t;
				if (cx >= 0 && cx < evensize && cy >= 0 && cy < evensize) {
					t=this.gridAt([cx,cy]).t;
				} else {
					t=this.bgter;
				}
				this.fillGridAt([x,y], { t: t });
			}
		}
		// Generate finer details until we reach the unit scale.
		if (step > 1) {
			this.worldFractal(evensize, step>>1);
		}
	}

	genDungeon() {
		if ( ! this.options) {
			this.options={
				iterations: this.density * Math.floor(this.size[0]/2) * Math.floor(this.size[1]/2),
				openDoor: 10,
				closedRoom: 40,
				openRoom: 20,
				semiOpenRoom: 10,
				doorHall: 20,
				openHall: 10,
				stripHooks: true,
				stripDeadDoors: true,
			};
		}

		let totalChance=0;
		totalChance += this.options.closedRoom;
		const closedRoom=totalChance;
		totalChance += this.options.openRoom;
		const openRoom=totalChance;
		totalChance += this.options.semiOpenRoom;
		const semiOpenRoom=totalChance;
		totalChance += this.options.doorHall;
		const doorHall=totalChance;
		totalChance += this.options.openHall;
		const openHall=totalChance;

		this.fillRect(rectInset(this.bounds(), 1), { t: "unknown" });
		this.drawRect(this.bounds(), { t: this.fgter });

		let feature={
			pt: [Math.floor(this.size[0]/4) + rnd(Math.floor(this.size[0]/2))+1,
				Math.floor(this.size[1]/4) + rnd(Math.floor(this.size[1]/2))+1 ],
			dir: rnd(Dir.count),
		};
		if ( ! this.dungeonOpenRoom(feature)) {
			throw new GridMapError("No space for starting room "+JSON.stringify(feature));
		}
		this.startPt=feature.pt;
		//DLOG("start "+JSON.stringify(feature));

		for (let i=0; i < this.options.iterations; ++i) {
			//DLOG("iter "+i);
			feature=this.dungeonSelectHook();
			if ( ! feature) {
				// no blank space available, stop mapping
				//DLOG("    no hook");
				break;
			}
			let rc=false;
			for (let j=0; j < this.options.iterations && ! rc; ++j) {
				const roll=rnd(totalChance)+1;
				if (roll <= closedRoom) {
					rc=this.dungeonClosedRoom(feature);
				} else if (roll <= openRoom) {
					rc=this.dungeonOpenRoom(feature);
				} else if (roll <= semiOpenRoom) {
					rc=this.dungeonSemiOpenRoom(feature);
				} else if (roll <= doorHall) {
					rc=this.dungeonDoorHall(feature);
				} else if (roll <= openHall) {
					rc=this.dungeonOpenHall(feature);
				}
			}
			if (rc) {
				//DLOG("    feature "+JSON.stringify(feature));
			} else {
				// if ITERATIONS tries didn't work, you can't put anything at that hook.
				//DLOG("    no feature");
				this.fillGridAt(feature.pt, { t: this.fgter });
			}
		}
		// cleanup
		this.drawRect(this.bounds(), { t: this.fgter });

		if (this.options.stripHooks) {
			this.filter(this.bounds(), (map, pt, g)=>{
				if (this.dungeonIsBuildableSpace(pt)) {
					let count=0;
					for (let d=0; d < Dir.count; ++d) {
						if (this.dungeonIsBuildableSpace(pointAdd(pt, Dir.delta[d]))) {
							++count;
						}
					}
					if (count <= 1) {
						// 1 bordering hook or unknown = wall
						this.fillGridAt(pt, { t: this.fgter });
					} else {
						this.fillGridAt(pt, { t: this.bgter });
					}
				}
			});
		}
		if (this.options.stripDeadDoors) {
			// Remove all doors which do not have exactly 2 bordering walls.
			this.filter(this.bounds(), (map, pt, g)=>{
				switch (g.t) {
					case "door":
					case "door_open":
					case "bars":
					case "bars_open": {
						const dirOpen=[];
						let count=0;
						let doordoor=false;
						for (let d=0; d < Dir.count; ++d) {
							const g2=this.gridAt(pointAdd(pt, Dir.delta[d]));
							if (g2.t === this.bgter) {
								dirOpen.push(true);
								++count;
							} else if (g2.t === "door" || g2.t === "door_open" || g2.t === "bars" || g2.t === "bars_open") {
								doordoor=true;
								dirOpen.push(false);
							} else {
								dirOpen.push(false);
							}
						}
						if ( ! doordoor && count === 2 &&
							((dirOpen[Dir.N] && dirOpen[Dir.S]) ||
							(dirOpen[Dir.E] && dirOpen[Dir.W])) ) {
							// nice door
						} else {
							g.t=this.bgter;
						}
					}
				}
			});
		}
	}

	/** Returns the hook for a feature, which is { pt: coord, dir: direction } */
	dungeonSelectHook() {
		const hooks=[];
		this.filter(this.bounds(), (map, pt, g)=>{
			switch (g.t) {
				case "hook_n":
					hooks.push( { pt: pt, dir: Dir.N } );
					break;
				case "hook_e":
					hooks.push( { pt: pt, dir: Dir.E } );
					break;
				case "hook_s":
					hooks.push( { pt: pt, dir: Dir.S } );
					break;
				case "hook_w":
					hooks.push( { pt: pt, dir: Dir.W } );
					break;
			}
		});
		return rndArrayChoose(hooks);
	}

	// computes feature.rect from pt, dir, size
	dungeonFeatureRect(feature) {
		switch (feature.dir) {
			case Dir.N:
				feature.rect=[feature.pt[0], feature.pt[1]-feature.size[1], feature.size[0], feature.size[1] ];
				break;
			case Dir.E:
				feature.rect=[feature.pt[0]+1, feature.pt[1], feature.size[0], feature.size[1] ];
				break;
			case Dir.S:
				feature.rect=[feature.pt[0], feature.pt[1]+1, feature.size[0], feature.size[1] ];
				break;
			case Dir.W:
				feature.rect=[feature.pt[0]-feature.size[0], feature.pt[1], feature.size[0], feature.size[1] ];
				break;
		}
	}

	/** can build over floor & hooks. */
	dungeonIsBuildableSpace(pt) {
		const g=this.gridAt(pt);
		return (g && (g.t === "unknown" || g.t === "hook_n" || g.t === "hook_e" || g.t === "hook_s" || g.t === "hook_w"));
	}

	dungeonCheckSpace(feature) {
		let check=true;
		this.filter(feature.rect, (map, pt, g)=>{
			if ( ! this.dungeonIsBuildableSpace(pt)) {
				check=false;
			}
		});
		return check;
	}

	dungeonClosedRoom(feature) {
		feature.name="closedRoom";
		feature.size=[rndDice(2, 3)+1, rndDice(2, 3)+1]; // was 2d6+1
		this.dungeonFeatureRect(feature);
		if ( ! this.dungeonCheckSpace(feature)) {
			return false;
		}
		this.fillRect(feature.rect, { t: this.bgter });
		this.drawRect(feature.rect, { t: this.fgter });
		for (let d=0; d < Dir.count; ++d) {
			let doorpt=null;
			switch (d) {
				case Dir.N:
					doorpt=[feature.rect[0] + rnd(feature.rect[2]-2)+1,
						feature.rect[1]];
					break;
				case Dir.E:
					doorpt=[feature.rect[0]+feature.rect[2],
						feature.rect[1] + rnd(feature.rect[3]-2)+1];
					break;
				case Dir.S:
					doorpt=[feature.rect[0] + rnd(feature.rect[2]-2)+1,
						feature.rect[1]+feature.rect[3]];
					break;
				case Dir.W:
					doorpt=[feature.rect[0],
						feature.rect[1] + rnd(feature.rect[3]-2)+1];
					break;
			}
			this.fillGridAt(doorpt, { t: this.randomDoorType() });
			const doorpt2=pointAdd(doorpt, Dir.delta[d]);
			if (this.dungeonIsBuildableSpace(doorpt2)) {
				this.fillGridAt(doorpt, { t: kHookForDir[d] });
			}
		}
		this.fillGridAt(feature.pt, [this.randomDoorType()]);

		const back=(feature.dir+2) % Dir.count;
		const backpt=pointAdd(feature.pt, Dir.delta[back]);
		if (this.dungeonIsBuildableSpace(backpt)) {
			this.fillGridAt(backpt, { t: kHookForDir[back] });
		}
	}

	randomDoorType() {
		const roll=rnd(100)+1;
		if (roll <= 20) { // locked
			return rnd(100)+1 <= this.options.openDoor ? "bars_open" : "bars";
		} else {
			return rnd(100)+1 <= this.options.openDoor ? "door_open" : "door";
		}
	}

	dungeonOpenRoom(feature) {
		feature.name="openRoom";
		feature.size=[rndDice(2, 3)+1, rndDice(2, 3)+1]; // was 2d6+1
		this.dungeonFeatureRect(feature);
		if ( ! this.dungeonCheckSpace(feature)) {
			return false;
		}
		this.fillRect(feature.rect, { t: this.bgter });
		for (let y=feature.rect[1]+1; y < feature.rect[1]+feature.rect[3]; ++y) {
			let x=feature.rect[0]-1;
			if (this.dungeonIsBuildableSpace([x,y])) {
				this.fillGridAt([x,y], { t: "hook_w" });
			}
			x=feature.rect[0] + feature.rect[2];
			if (this.dungeonIsBuildableSpace([x,y])) {
				this.fillGridAt([x,y], { t: "hook_e" });
			}
		}
		for (let x=feature.rect[0]+1; x < feature.rect[0]+feature.rect[2]; ++x) {
			let y=feature.rect[1]-1;
			if (this.dungeonIsBuildableSpace([x,y])) {
				this.fillGridAt([x,y], { t: "hook_n" });
			}
			y=feature.rect[1] + feature.rect[3];
			if (this.dungeonIsBuildableSpace([x,y])) {
				this.fillGridAt([x,y], { t: "hook_s" });
			}
		}
		this.fillGridAt(feature.pt, { t: this.bgter });
		const back=(feature.dir + 2) % Dir.count;
		this.fillGridAt(pointAdd(feature.pt, Dir.delta[back]), { t: this.bgter });
		return true;
	}

	dungeonSemiOpenRoom(feature) {
		// FIXME:
		return this.dungeonOpenRoom(feature);
	}

	dungeonDoorHall(feature) {
		feature.name="doorHall";
		if (feature.dir === Dir.N || feature.dir === Dir.S) {
			feature.size=[1, rnd(6)+3];
		} else {
			feature.size=[rnd(6)+3, 1];
		}
		this.dungeonFeatureRect(feature);
		if ( ! this.dungeonCheckSpace(feature)) {
			return false;
		}
		this.fillRect(feature.rect, { t: this.bgter });
		this.fillGridAt(feature.pt, { t: this.randomDoorType() });
		this.dungeonAddTail(feature);
		return true;
	}

	dungeonOpenHall(feature) {
		feature.name="openHall";
		if (feature.dir === Dir.N || feature.dir === Dir.S) {
			feature.size=[1, rnd(6)+3];
		} else {
			feature.size=[rnd(6)+3, 1];
		}
		this.dungeonFeatureRect(feature);
		if ( ! this.dungeonCheckSpace(feature)) {
			return false;
		}
		this.fillRect(feature.rect, { t: this.bgter });
		this.fillGridAt(feature.pt, { t: this.bgter });
		const back=(feature.dir+2) % Dir.count;
		const backpt=pointAdd(feature.pt, Dir.delta[back]);
		if (this.dungeonIsBuildableSpace(backpt)) {
			this.fillGridAt(backpt, { t: this.bgter });
		}
		this.dungeonAddTail(feature);
		return true;
	}

	dungeonAddTail(feature) {
		const pt=[feature.pt[0] + feature.size[0] * Dir.delta[feature.dir][0],
			feature.pt[1] + feature.size[1] * Dir.delta[feature.dir][1] ];
		const back=(feature.dir+2) % Dir.count;
		for (let d=0; d < Dir.count; ++d) {
			// don't build backwards
			if (d !== back) {
				const pt2=pointAdd(pt, Dir.delta[d]);
				if (this.dungeonIsBuildableSpace(pt2)) {
					this.fillGridAt(pt2, { t: kHookForDir[d] });
				}
			}
		}
	}

	printWalkable(blockingFunc) {
		let s="";
		this.filter(this.bounds(), (map, pt, g)=>{
			if (blockingFunc(g)) {
				s+="#";
			} else if (this.gridFlagTest(g, kGridFlag.walkable)) {
				s+=".";
			} else {
				s+="?";
			}
			if (pt[0] === this.size[0]-1) {
				s+="\n";
			}
		});
		console.log("Walkable map:\n"+s);
	}

	/** Sets walkable flag on all grids attached to enterPt, which are not blockingFunc(g). */
	floodFillWalkable(enterPt, blockingFunc) {
		this.gridFlagSet(this.gridAt(enterPt), kGridFlag.walkable);
		let any;
		do {
			any=false;
			this.filter(this.bounds(), (map, pt, g)=>{
				if ( ! blockingFunc(g) && this.gridFlagTest(g, kGridFlag.walkable)) {
					for (let dir=0; dir < Dir.count; ++dir) {
						const pt2=pointAdd(pt, Dir.delta[dir]);
						const g2=this.gridAt(pt2);
						if (g2 && ! blockingFunc(g2) && ! this.gridFlagTest(g2, kGridFlag.walkable)) {
							this.gridFlagSet(g2, kGridFlag.walkable);
							any=true;
						}
					}
				}
			});
			//this.printWalkable(blockingFunc);
		} while (any);
	}

	/** For a given point which is not walkable, find the nearest one which is,
	* draw a line to it in fillg, for example {t:"floor"}.
	* Returns point reached by escape, or null if it failed.
	*/
	escapeUnreachable(pt, blockingFunc, fillg) {
		const deltas=[];
		for (let dir=0; dir < Dir.count; ++dir) {
			deltas.push(Dir.delta[dir]);
		}
		rndArrayShuffle(deltas); // so each block doesn't escape north preferentially
		for (let dist=1, maxdist=Math.max(this.size[0], this.size[1]); dist < maxdist; ++dist) {
			for (const delta of deltas) {
				const pt1=pointAdd(pt, pointMult(delta, dist));
				const g1=this.gridAt(pt1);
				if (g1 && this.gridFlagTest(g1, kGridFlag.walkable)) {
					const points=pointsOnLine(pt, pt1);
					//DLOG("    escaping "+pt+" by "+delta+" to "+pt1+": "+points.join(" ; "));
					for (const pt2 of points) {
						const g2=this.gridAt(pt2);
						if (g2 && blockingFunc(g2)) {
							this.fillGridAt(pt2, fillg);
							this.gridFlagSet(g2, kGridFlag.walkable);
						}
					}
					return pt1;
				}
			}
		}
		throw new GridMapError("Unable to escape unreachable "+pt);
		return null;
	}

} // class TileGridMap

