/* mdhutil.js
 * Cyberhole
 * Created 2025-07-11
 * Copyright © 2025 by Mark Damon Hughes. All Rights Reserved.
 * See BSDLicense.txt
 */

/* eslint-env browser, es2020 */

"use strict";

//---------------------------------------
// Utility functions

let DEBUG=true;

function $(id) { return document.getElementById(id); }

/** Logs to the console only if DEBUG is true. */
function DLOG() {
	if (DEBUG) {
		console.log.apply(this, arguments); // eslint-disable-line prefer-rest-params
	}
}

/** Translate plain text into legal HTML. */
function htmlify(text) {
	return String(text).replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/\n/g, "<br>");
}

function jsonToFormEncoded(data) {
	const body=[];
	for (const k in data) {
		body.push(encodeURIComponent(k) + "=" + encodeURIComponent(data[k]) );
	}
	return body.join("&");
}

/** Requests a document from a url,
* data is a simple object with key:value pairs.
* callback receives (result, null) or (null, errorMessage).
* result is not parsed, do that in client code.
*/
function ajax(method, url, data, callback) {
	const datatext=data ? jsonToFormEncoded(data) : null;
	const request=new XMLHttpRequest();
	//request.withCredentials=true;
	if ( ! request) {
		const msg="No XMLHttpRequest";
		output("red", msg);
		return false;
	}
	request.onreadystatechange=()=>{
		//console.log("ajax state "+request.readyState+", cookies=", document.cookie);
		if (request.readyState !== XMLHttpRequest.DONE) {
			return false;
		} else if ((request.status === 200 || request.status === 0) &&
				! request.responseText.startsWith("#!")) {
			return callback(request.responseText, null);
		} else {
			return callback(null, url+" returned "+request.status+
				" "+request.statusText+"\n"+request.responseText);
		}
	};
	if (data && method === "GET") {
		url += "?"+datatext;
	}
	request.open(method, url, true);
	if (data && method !== "GET") {
		request.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
		request.send(datatext);
	} else {
		request.send();
	}
}

/** Returns string value of a specific cookie, or defaultValue (null). */
function htmlCookieGet(name, defaultValue=null) {
	const cname=name+"=";
	const allcookies=decodeURIComponent(document.cookie).split(/; */);
	for (const c of allcookies) {
		if (c.indexOf(cname) === 0) {
			return c.substring(cname.length);
		}
	}
	return defaultValue;
}

/** Sets a specific cookie to string of value, or null. */
function htmlCookieSet(name, value, exdays=31) {
	const d=new Date();
	d.setTime(d.getTime() + exdays*24*60*60*1000);
	document.cookie=name+"="+value+";expires="+d.toUTCString()+";path=/";
	// is UTCString correct?
}

/** Returns true if localStorage is available. */
function htmlStorageAvailable() {
	try {
		const testValue="_test";
		const storage=window["localStorage"];
		localStorage[testValue]=testValue;
		const s=storage[testValue];
		storage.removeItem(testValue);
		if (s === testValue) {
			return true;
		} else {
			console.error("localStorage failed: "+s);
			return false;
		}
	} catch (e) {
		console.error("localStorage not available: "+e);
		return false;
	}
}

/** Returns value for `key` from localStorage,
 * or `defaultValue` (default null) if it was not stored.
 **/
function htmlStorageGet(key, defaultValue=null) {
	const text=localStorage[key];
	if (text && text !== "undefined") {
		try {
			return JSON.parse(text);
		} catch (e) {
			console.error("Error storageGet '"+key+"':", text, e);
			return defaultValue;
		}
	} else {
		return defaultValue;
	}
}

/** Saves `data` for `key` in localStorage.
* data must be JSON types only.
*/
function htmlStorageSet(key, data) {
	try {
		const text=JSON.stringify(data);
		localStorage[key]=text;
		return true;
	} catch (e) {
		console.error("Error storageSet '"+key+"'", data, e);
		return false;
	}
}

/** Return a new array `len` long, filled with `value`. */
function arrayDim(len, value=0) {
	const a=Array(len);
	a.fill(value);
	return a;
}

/** Returns a multi-dimensional array, filled with `value`.
 * dims=[2,3,4] creates
 * [ [ [0,0,0,0], [0,0,0,0], [0,0,0,0] ],
 *   [ [0,0,0,0], [0,0,0,0], [0,0,0,0] ],
 * ]
 */
function arrayDimMulti(dims, value=0) {
	const n=arrayFirst(dims);
	const arr=arrayDim(n, value);
	if (dims.length>1) {
		const subdims=dims.slice(1);
		for (let i=0; i<n; ++i) {
			arr[i] = arrayDimMulti(subdims, value);
		}
	}
	return arr;
}

/** Reach into a multi-dimension array `arr` at position `dims`,
 * and return the value.
 */
function arrayGetMulti(arr, dims) {
	const n=arrayFirst(dims);
	if (n<0 || n>=arr.length) {
		throw new Error("get index out of bounds "+n+"/"+arr.length);
	}
	if (dims.length===1) {
		return arr[n];
	} else {
		const subdims=dims.slice(1);
		const subarr=arr[n];
		return arrayGetMulti(arr[n], subdims);
	}
}

/** Reach into a multi-dimension array `arr` at position `dims`,
 * and set it to value.
 */
function arraySetMulti(arr, dims, value=0) {
	const n=arrayFirst(dims);
	if (n<0 || n>=arr.length) {
		throw new Error("set index out of bounds "+n+"/"+arr.length);
	}
	if (dims.length===1) {
		arr[n]=value;
	} else {
		const subdims=dims.slice(1);
		arraySetMulti(arr[n], subdims, value);
	}
}

/** Returns first element in array, or null if empty array. */
function arrayFirst(arr) {
	return arr.length >= 1 ? arr[0] : null;
}

/** Adds `n` (default 1) to `arr[key]`, defaulting to 0. */
function arrayHistogram(arr, key, n=1) {
	arr[key]=(arr[key]||0) + n;
}

/** Removes 'value' from array and returns it if it was found, otherwise returns null. */
function arrayRemove(arr, value) {
	const i=arr.indexOf(value);
	if (i >= 0) {
		arr.splice(i, 1);
		return value;
	}
	return null;
}

/** Generic comparison for sorting.
* arr.sort((a,b)=>{ return compare(a.name, b.name); });
*/
function compare(a, b) {
	if (isArray(a) && isArray(b)) {
		for (let i=0, minlen=Math.min(a.length, b.length); i < minlen; ++i) {
			if (a[i]<b[i]) {
				return -1;
			} else if (a[i]>b[i]) {
				return 1;
			}
		}
		return compare(a.length, b.length);
	}
	// unit compare
	if (a<b) {
		return -1;
	} else if (a>b) {
		return 1;
	} else {
		return 0;
	}
}

function isNull(x) { return x===undefined || x===null || Number.isNaN(x); }
function isNumber(x) { return Number.isFinite(x); }
function isString(x) { return typeof(x)==="string"; }
function isArray(x) { return Array.isArray(x); }
function isDigit(c) { return c>='0' && c<='9'; }
function isAlpha(c) { return (c>='A' && c<='Z') || (c>='a' && c<='z'); }
function isWhitespace(c) { return c===' ' || c==='\t' || c==='\n' || c==='\r'; }

/** Returns integer for any number or string, 0 if it can't be parsed. */
function toInt(x) {
	let y;
	if (isNumber(x)) {
		y=x;
	} else if (isString(x)) {
		y=parseInt(x,10);
		if (!isNumber(y)) {
			y=0;
		}
	} else {
		y=0;
	}
	return Math.floor(y);
}

/** Clips x to range low..hi. */
function minmax(x, low, hi) {
	return Math.min(Math.max(low,x),hi);
}

/** For an object where `obj[key] := row`,
 * returns an object where `row[invkey] := key`.
 * Used to make a reverse index of an object.
 **/
function objectInvert(obj, invkey) {
	const t={};
	for (let k in obj) {
		const row=obj[k];
		t[row[invkey]]=k;
	}
	return t;
}

/** Creates an object from a sequence of key-value pairs.
 * Example: objectMake("t"," ", Dir.N,0, Dir.E,0, Dir.S,0, Dir.W,0)
 **/
function objectMake() {
	const obj={};
	for (let i=0; i<arguments.length; i+=2) {
		obj[arguments[i]]=arguments[i+1];
	}
	return obj;
}

function randomToken() {
	if (self.crypto.randomUUID) {
		return self.crypto.randomUUID();
	} else {
		return Date.now()+""+Math.random();
	}
}

function rndFloat() {
	return Math.random();
}

/** Returns a random integer from 0 to n-1. */
function rnd(n) {
	return Math.floor(rndFloat() * n);
}

/** Choose one item at random from array `arr`, null if empty array. */
function rndArrayChoose(arr) {
	if (arr.length===0) {
		return null;
	}
	const i=rnd(arr.length);
	return arr[i];
}

/** Fisher-Yates */
function rndArrayShuffle(arr) {
	for (let i=arr.length-1; i>=1; --i) {
		const j=rnd(i+1);
		const tmp=arr[i];
		arr[i]=arr[j];
		arr[j]=tmp;
	}
	return arr;
}

/** Returns the roll of 'n' dice of 's' sides, e.g. dice(3, 6) returns 3-18.
* Records all rolls in array diceRolled, if given.
*/
function rndDice(n, s, diceRolled=null) {
	n=Math.ceil(n); s=Math.ceil(s);
	if (n<=0 || s<=0) {
		return 0;
	}
	let t=0;
	for (let i=0; i<n; ++i) {
		let r=rnd(s) + 1;
		t += r;
		if (diceRolled !== null) {
			diceRolled.push(r);
		}
	}
	return t;
}

/** Choose from `table` array of [chance, value] entries,
* rolls against total chance if `roll` is not given.
*/
function diceTable(table, roll=null) {
	let t=0;
	for (let i=0; i<table.length; ++i) {
		t += table[i][0];
	}
	if (t<=0) {
		console.error("Invalid dice table "+table);
		return null;
	}
	if ( ! roll) {
		roll=rnd(t)+1;
	}
	let c=0;
	for (let i=0; i<table.length; ++i) {
		c += table[i][0];
		if (roll<=c) {
			return table[i][1];
		}
	}
	console.error("Invalid roll "+roll+" on diceTable "+table);
	return null;
}

/** Returns the largest power of 2 which is <= n */
function roundPowerOf2(n) {
	let pos=0;
	while (n>0) {
		++pos;
		n>>=1;
	}
	return 1<<(pos-1);
}

/** Pads start and end of obj.toString() with padchars
 * (default ' ', you may prefer '\xa0' nbsp) so it fills n chars;
 * odd char is added at the end.
 **/
function strPadCenter(obj, n, padchars=' ') {
	const s=String(obj), slen=s.length;
	const left=Math.floor( (n-slen)/2), right=Math.ceil( (n-s.length)/2);
	let out="";
	for (let i=0; i<left; ++i) {
		out+=padchars;
	}
	out+=s;
	for (let i=0; i<right; ++i) {
		out+=padchars;
	}
	return out;
}

/** Pads start of obj.toString() with padchars so it fills n chars.
 * default ' ', you may prefer '\xa0' nbsp, or '0' for numbers.
 */
function strPadLeft(obj, n, padchars=' ') {
	return String(obj).padStart(n, padchars);
}

/** Pads end of obj.toString() with padchars so it fills n chars.
 * default ' ', you may prefer '\xa0' nbsp, or '0' for numbers.
 */
function strPadRight(obj, n, padchars=' ') {
	return String(obj).padEnd(n, padchars);
}

/** Provides a very simplistic English plural of 'n' units of 's'.
* 1 plum, 2 plums, 1 quartz, 2 quartzes, 1 bunny, 2 bunnies.
*/
function strPlural(n, s) {
	let text=""+n+" "+s;
	if (n!==1) {
		if (s.endsWith("y")) {
			text=text.substr(0, text.length-1)+"ies";
		} else if (s.endsWith("x") || s.endsWith("z")) {
			text+="es";
		} else {
			text+="s";
		}
	}
	return text;
}

/** Return string `c` repeated `n` times. */
function strRep(c, n) {
	return "".padEnd(n, c);
}

/** Splits lines in `s` at any newlines, DOS, Mac, or Unix.
 * If `stripblank` is true, removes empty lines.
 **/
function strSplitLines(s, stripblank=false) {
	let lines=s.split(/\r?\n|\r|\n/g);
	if (stripblank) {
		lines=lines.filter( (s)=>{ return s.length>0; } );
	}
	return lines;
}

/** Splits words in `s` at any whitespace.
 * If `stripblank` is true, removes empty words.
 **/
function strSplitWords(s, stripblank=false) {
	let words=s.split(/\s+/);
	if (stripblank) {
		words=words.filter((s)=>{ return s.length>0; });
	}
	return words;
}

const MDH_FPS=60;
const MDH_FRAME_TIME=Math.floor(1000/MDH_FPS);

class Timer {
	constructor(updater, stopper, time=MDH_FRAME_TIME) {
		this.timestamp=0;
		this.updater=updater;
		this.stopper=stopper;
		this.time=time;
		this.start();
	}

	start() {
		this.timer=setInterval(()=>{ this.updater(Date.now()); }, this.time);
	}

	/** Cancel the timer. You should do this on stopping the application, but
	 * also whenever it loses focus and shouldn't be doing anything.
	 */
	stop() {
		if (this.timer) {
			clearInterval(this.timer);
			this.timer=null;
		}
		this.stopper();
	}

} // Timer

