/* tbasicweb.js
 * Cyberhole
 * Created 2025-07-11
 * Copyright © 2025 by Mark Damon Hughes. All Rights Reserved.
 * See BSDLicense.txt
 */

// TODO: type line number to delete it
// TODO: input multiple variables

// requires mdhutil.js, terminal.js

/* eslint-env browser, es2020 */

"use strict";

const TBW_VERSION="TinyBasicWeb 0.9";

//---------------------------------------
// TinyBasicWeb

const TBW_LINENUM_MAX=32767;
const TBW_STRING_MAX=255;
const TBW_DIM_MAX=65536;
const TBW_FILENAME_STRING="[A-Za-z0-9][A-Za-z0-9_.-]{0,15}";
const TBW_FILENAME_PATTERN=RegExp("^"+TBW_FILENAME_STRING+"$");
const TBW_DIR_PATTERN=RegExp(".*<a href=\"("+TBW_FILENAME_STRING+")\">.*$");

// states map messages, mostly keypresses, to events.
// "any" receives any undefined characters, plain text key.

const TBW_STATE_RUN={
	"F1":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_CMD; return true; } },
	"F2":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_RUN; return true; } },
	"F3":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_MEMO; return true; } },
	"F4":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_INPUT; return true; } },
	"any":			(key)=>{ tbw.setInkey(key); return true; },
	"^z":			(key)=>{ tbw.breakKey(); return true; },
};

const TBW_STATE_WAIT={
	// TODO: ^z=end waiting task, break
	"any":			(key)=>{ tbw.setInkey(key); return true; },
	"F1":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_CMD; return true; } },
	"F2":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_RUN; return true; } },
	"F3":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_MEMO; return true; } },
	"F4":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_INPUT; return true; } },
	"^z":			(key)=>{ tbw.breakKey(); return true; },
};

const TBW_STATE_CMD={
	"Enter":		(key)=>{ tbw.commandEnter(); return true; },
	"Backspace":	(key)=>{ terminal.print("\b"); return true; },
	"Delete":		(key)=>{ terminal.delforward(); return true; },
	"Help":			(key)=>{ terminal.insert(); return true; },
	"Insert":		(key)=>{ terminal.insert(); return true; },
	"ArrowLeft":	(key)=>{ terminal.cursorMoveBy([-1,0]); return true; },
	"ArrowRight":	(key)=>{ terminal.cursorMoveBy([1,0]); return true; },
	"ArrowUp":		(key)=>{ terminal.cursorMoveBy([0,-1]); return true; },
	"ArrowDown":	(key)=>{ terminal.cursorMoveBy([0,1]); return true; },
	"Home":			(key)=>{ terminal.cursorHome(); return true; },
	"End":			(key)=>{ terminal.cursorEnd(); return true; },
	"Tab":			(key)=>{ terminal.print("\t"); return true; },
	"any":			(key)=>{ if (terminal.cursor()[0]<terminal.termsize()[0]-1) {
								terminal.print(key); return true;
							} else { return false; }
					},
	"F1":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_CMD; return true; } },
	"F2":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_RUN; return true; } },
	"F3":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_MEMO; return true; } },
	"F4":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_INPUT; return true; } },
	"^z":			(key)=>{ tbw.breakKey(); return true; },
};

const TBW_STATE_MEMO={
	"Enter":		(key)=>{ terminal.print("\n"); return true; },
	"Backspace":	(key)=>{ terminal.print("\b"); return true; },
	"Delete":		(key)=>{ terminal.delforward(); return true; },
	"Help":			(key)=>{ terminal.insert(); return true; },
	"Insert":		(key)=>{ terminal.insert(); return true; },
	"ArrowUp":		(key)=>{ terminal.cursorMoveBy([0,-1]); return true; },
	"ArrowDown":	(key)=>{ terminal.cursorMoveBy([0,1]); return true; },
	"ArrowLeft":	(key)=>{ terminal.cursorMoveBy([-1,0]); return true; },
	"ArrowRight":	(key)=>{ terminal.cursorMoveBy([1,0]); return true; },
	"Tab":			(key)=>{ terminal.print("\t"); return true; },
	"Home":			(key)=>{ terminal.cursorHome(); return true; },
	"End":			(key)=>{ terminal.cursorEnd(); return true; },
	"PageUp":		(key)=>{ terminal.cursorPageUp(); return true; },
	"PageDown":		(key)=>{ terminal.cursorPageDown(); return true; },
	"any":			(key)=>{ terminal.print(key); return true; },
	"F1":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_RUN; return true; } },
	"F2":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_CMD; return true; } },
	"F3":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_MEMO; return true; } },
	"F4":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_INPUT; return true; } },
	"^z":			(key)=>{ tbw.breakKey(); return true; },
};

const TBW_STATE_INPUT={
	"Enter":		(key)=>{ tbw.inputEnter(); return true; },
	"Backspace":	(key)=>{
		if (terminal._cursor[0]>tbw.inputLeft) {
			terminal.print("\b");
			return true;
		} else {
			return false;
		}
	},
	"Delete":		(key)=>{ terminal.delforward(); return true; },
	"Help":			(key)=>{ terminal.insert(); return true; },
	"Insert":		(key)=>{ terminal.insert(); return true; },
	"ArrowLeft":	(key)=>{
		if (terminal._cursor[0]>tbw.inputLeft) {
			terminal.cursorMoveBy([-1,0]);
			return true;
		} else {
			return false;
		}
	},
	"ArrowRight":	(key)=>{
		if (terminal._cursor[0]<terminal.termsize()[0]-1) {
			terminal.cursorMoveBy([1,0]);
			return true;
		} else {
			return false;
		}
	},
	"Home":			(key)=>{
		terminal.setCursor([tbw.inputLeft, terminal.cursor()[1]]);
		return true;
	},
	"End":			(key)=>{ terminal.cursorEnd(); return true; },
	"Tab":			(key)=>{ terminal.print("\t"); return true; },
	"any":			(key)=>{
		if (terminal._cursor[0]<terminal.termsize()[0]-1) {
			terminal.print(key);
			return true;
		} else {
			return false;
		}
	},
	"F1":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_CMD; return true; } },
	"F2":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_RUN; return true; } },
	"F3":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_MEMO; return true; } },
	"F4":			(key)=>{ if (DEBUG) { tbw.state=TBW_STATE_INPUT; return true; } },
	"^z":			(key)=>{ tbw.breakKey(); return true; },
};

const TBW_FUNCTIONS = {
	"ABS":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected ABS(n) at line "+tbw.pc);
					}
					return Math.abs(args[0]);
				},
	"ASC":		(args)=>{
					if (args.length!==1 || !isString(args[0])) {
						throw new BasicError("Expected ASC(a$) at line "+tbw.pc);
					}
					return args[0].length===0 ? 0 : args[0].codePointAt(0);
				},
	"AT":		(args)=>{
					if (args.length!==2 || !isNumber(args[0]) || !isNumber(args[1])) {
						throw new BasicError("Expected AT(x,y) at line "+tbw.pc);
					}
					let x=minmax(Math.floor(args[0]), 0, terminal.termsize()[0]);
					let y=minmax(Math.floor(args[1]), 0, terminal.termsize()[1]);
					terminal.setCursor([x,y]);
					return "";
				},
	"COLOR":	(args)=>{
					if (args.length!==2 || !isNumber(args[0]) || !isNumber(args[1])) {
						throw new BasicError("Expected COLOR(fg,bg) at line "+tbw.pc);
					}
					terminal.fgcolor=minmax(Math.floor(args[0]), 0, TERMCOLORMAX);
					terminal.bgcolor=minmax(Math.floor(args[1]), 0, TERMCOLORMAX);
					return "";
				},
	"ATN":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected ATN(r) at line "+tbw.pc);
					}
					return Math.atan(args[0]);
				},
	"CHR$":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected CHR$(n) at line "+tbw.pc);
					}
					return String.fromCodePoint(Math.floor(args[0]));
				},
	"COS":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected COS(r) at line "+tbw.pc);
					}
					return Math.cos(args[0]);
				},
	"DATE$":	(args)=>{
					if (args.length!==0) {
						throw new BasicError("Expected DATE$() at line "+tbw.pc);
					}
					let s=new Date().toISOString();
					return s.substring(0,10);
				},
	"EXP":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected EXP(n) at line "+tbw.pc);
					}
					return Math.exp(args[0]);
				},
	"INKEY$":	(args)=>{
					if (args.length!==0) {
						throw new BasicError("Expected INKEY$() at line "+tbw.pc);
					}
					let c=tbw.inkey || "";
					tbw.inkey=null;
					return c;
				},
	"INSTR":	(args)=>{
					if (args.length<2 || args.length>3 ||
						!isString(args[0]) || !isString(args[1]) ||
						(args.length===3 && !isNumber(args[2])) ) {
						throw new BasicError("Expected INSTR(n$,h$,x) at line "+tbw.pc);
					}
					let h=args[0], n=args[1];
					let x=minmax((args.length===3 ? Math.floor(args[2]) : 0), 1, h.length);
					if (h==="" || n==="") {
						return 0;
					}
					return h.indexOf(n, x-1)+1;
				},

	"INT":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected INT(n) at line "+tbw.pc);
					}
					return Math.floor(args[0]);
				},
	"LCASE$":	(args)=>{
					if (args.length!==1 || !isString(args[0])) {
						throw new BasicError("Expected LCASE$(n) at line "+tbw.pc);
					}
					return args[0].toLocaleLowerCase();
				},
	"LEN":		(args)=>{
					if (args.length!==1 || !isString(args[0])) {
						throw new BasicError("Expected LEN(a$) at line "+tbw.pc);
					}
					return args[0].length;
				},
	"LOC":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected LOC(n) at line "+tbw.pc);
					}
					switch (Math.floor(args[0])) {
						case 0: return terminal.cursor()[0];
						case 1: return terminal.cursor()[1];
						case 2: return terminal.termsize()[0];
						case 3: return terminal.termsize()[1];
						case 4: { let cont=terminal.contentsAt(terminal.cursor());
							return cont ? cont[0].codePointAt(0) : 32;
						}
						case 5: { let cont=terminal.contentsAt(terminal.cursor());
							return cont && cont[1]!==null ? cont[1] : terminal.fgcolor;
						}
						case 6: { let cont=terminal.contentsAt(terminal.cursor());
							return cont && cont[2]!==null ? cont[2] : terminal.bgcolor;
						}
						default: return 0;
					}
				},
	"LOG":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected LOG(n) at line "+tbw.pc);
					}
					let rc=Math.log(args[0]);
					return isNumber(rc) ? rc : 0;
				},
	"MID$":		(args)=>{
					if (args.length!==3 || !isString(args[0]) || !isNumber(args[1]) || !isNumber(args[2])) {
						throw new BasicError("Expected MID$(a$,i,n) at line "+tbw.pc);
					}
					let a=args[0];
					let i=minmax(Math.floor(args[1])-1, 0, a.length-1);
					let n=minmax(Math.floor(args[2]), 0, a.length-i);
					return args[0].substring(i, i+n);
				},
	"PI":		(args)=>{
					if (args.length!==0) {
						throw new BasicError("Expected PI() at line "+tbw.pc);
					}
					return Math.PI;
				},
	"PIX$":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected PIX$(n) at line "+tbw.pc);
					}
					let n=minmax(Math.floor(args[0]), 0, 15);
					return TermPixels[n];
				},
	"RND":		(args)=>{
					if (args.length!==0) {
						throw new BasicError("Expected RND() at line "+tbw.pc);
					}
					return rndFloat();
				},
	"SIN":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected SIN(r) at line "+tbw.pc);
					}
					return Math.sin(args[0]);
				},

	"SQR":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected SQR(n) at line "+tbw.pc);
					}
					let rc=Math.sqrt(args[0]);
					return isNumber(rc) ? rc : 0;
				},
	"STR$":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected STR$(n) at line "+tbw.pc);
					}
					return args[0].toString();
				},
	"STRCMP":	(args)=>{
					if (args.length!==2 || !isString(args[0]) || !isString(args[1])) {
						throw new BasicError("Expected STRCMP(a$,b$) at line "+tbw.pc);
					}
					let a=args[0], b=args[1];
					return (a<b) ? -1 : (a>b) ? 1 : 0;
				},
	"TAB":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected TAB(x) at line "+tbw.pc);
					}
					let x=minmax(Math.floor(args[0]), 0, terminal.termsize()[0]);
					let y=terminal.cursor()[1];
					terminal.setCursor([x,y]);
					return "";
				},
	"TAN":		(args)=>{
					if (args.length!==1 || !isNumber(args[0])) {
						throw new BasicError("Expected TAN(r) at line "+tbw.pc);
					}
					return Math.tan(args[0]);
				},
	"TIME$":	(args)=>{
					if (args.length!==0) {
						throw new BasicError("Expected TIME$() at line "+tbw.pc);
					}
					let s=new Date().toISOString();
					return s.substring(11,s.length-1);
				},
	"TIMER":	(args)=>{
					if (args.length!==0) {
						throw new BasicError("Expected TIMER() at line "+tbw.pc);
					}
					return Date.now()/1000.0;
				},
	"UCASE$":	(args)=>{
					if (args.length!==1 || !isString(args[0])) {
						throw new BasicError("Expected UCASE$(n) at line "+tbw.pc);
					}
					return args[0].toLocaleUpperCase();
				},
	"VAL":		(args)=>{
					if (args.length!==1 || !isString(args[0])) {
						throw new BasicError("Expected VAL(a$) at line "+tbw.pc);
					}
					let n=parseFloat(args[0],10);
					return isNumber(n) ? n : 0;
				},
};

class BasicError extends Error { }

class BasicToken {
	constructor(name) {
		this.name=name;
	}
	classname() { return "Token"; }
	toList() {
		return this.name+" ";
	}
	toString() {
		if (DEBUG) {
			return "["+this.classname()+" "+this.name+"]";
		}
		return this.name;
	}
}

class BasicLabel extends BasicToken {
	constructor(name, num) {
		if (name) {
			super(name);
			this.num=null;
		} else {
			super(null);
			this.num=num;
		}
	}
	classname() { return "Label"; }
	toList() {
		return (this.num ? this.num : this.name);
	}
	toString() {
		if (DEBUG) {
			return "["+this.classname()+" "+this.name+","+this.num+"]";
		}
		return this.name ? this.name : this.num.toString();
	}
}

class BasicStmt extends BasicToken {
	classname() { return "Stmt"; }
	toList() {
		return " "+this.name+" ";
	}
}

class BasicString extends BasicToken {
	classname() { return "String"; }
	toList() {
		return '"'+this.name+'"';
	}
	toString() {
		if (DEBUG) {
			return '['+this.classname()+' "'+this.name+'"]';
		}
		return '"'+this.name+'"';
	}
}

class BasicNum extends BasicToken {
	constructor(num) {
		super(null);
		this.num=num;
	}
	classname() { return "Num"; }
	toList() {
		return this.num;
	}
	toString() {
		if (DEBUG) {
			return "["+this.classname()+" "+this.num+"]";
		}
		return this.num.toString();
	}
}

class BasicVar extends BasicToken {
	constructor(name, argList) {
		super(name);
		this.argList=argList;
	}
	classname() { return "Var"; }
	isString() {
		return this.name.endsWith('$');
	}
	toList() {
		let s=this.name;
		if (this.argList) {
			s+="(";
			let first=true;
			for (let a of this.argList) {
				if (first) {
					first=false;
				} else {
					s+=",";
				}
				s+=a.toList();
			}
			s+=")";
		}
		return s;
	}
	toString() {
		if (DEBUG) {
			return "["+this.classname()+" "+this.name+" "+this.argList+"]";
		}
		return this.name+
			(this.argList===null ? "" : "("+this.argList.join(",")+")");
	}
}

class BasicExpr extends BasicToken {
	constructor(argList) {
		super(null);
		this.argList=argList;
	}
	classname() { return "Expr"; }
	toList() {
		let s="(";
		if (this.argList) {
			for (let a of this.argList) {
				s+=a.toList();
			}
			s+=")";
		}
		return s;
	}
	toString() {
		if (DEBUG) {
			return "["+this.classname()+" "+this.argList+"]";
		}
		return "("+this.argList.join(",")+")";
	}
}

class BasicOp extends BasicToken {
	classname() { return "Op"; }
	toList() {
		return this.name;
	}
}

class BasicSep extends BasicToken {
	classname() { return "Sep"; }
	toList() {
		return this.name;
	}
}

class TinyBasicWeb {

	constructor() {
		this.doNew();

		// boot message
		terminal.bgcolor=TermColor.black;
		for (let x=1; x<=TERMCOLORMAX; ++x) {
			terminal.fgcolor=x;
			terminal.print(TermPixels[15]);
		}
		terminal.fgcolor=rnd(TERMCOLORMAX-1)+1;
		terminal.print(" "+TBW_VERSION+"\n");

		this.start();
	}

	start() {
		DLOG("TinyBasicWeb.start");
		this.pauseUntil=null;
		this.inkey=null;
		terminal.timer.updater=(timestamp)=>{ this.update(timestamp); };
		terminal.timer.stopper=()=>{
			this.acceptInput=false;
			this.acceptCursor=false;
		};
		this.readyPrompt();
	}

	readyPrompt() {
		terminal.bgcolor=TermColor.black; terminal.fgcolor=TermColor.cyan;
		terminal.print("READY\n");
		terminal.fgcolor=TermColor.white;
		this.state=TBW_STATE_CMD;
	}

	update(timestamp) {
		this.timestamp=timestamp;
		if (terminal) {
			while (terminal.keyQueue.length>0) {
				const c=terminal.keyQueue.shift();
				let ok=false;
				let k=this.state[c] || this.state["any"];
				DLOG("key "+c+": "+k);
				if (k) {
					ok=k(c);
				}
				if ( ! ok) {
					terminal.beep();
				}
			}
		}
		let t2=Date.now();
		while (this.state===TBW_STATE_RUN && t2 - timestamp<=MDH_FRAME_TIME/2) {
			//DLOG("run "+t2+" state="+this.state+" pauseUntil="+this.pauseUntil);
			if (this.pauseUntil!==null) {
				if (t2<this.pauseUntil) {
					break; // do nothing
				} else {
					this.pauseUntil=null;
				}
			}
			if (this.pc<=0 || this.pc>this.programEnd) {
				this.doEnd();
			} else if (this.program[this.pc]) {
				this.eval(this.program[this.pc], 0);
			} else {
				this.nextPC();
			}
			t2=Date.now();
		}
		//DLOG("run "+(t2-timestamp)+" terminal.needsRedraw="+terminal.needsRedraw() );
		if (terminal && (terminal.needsRedraw() || t2-terminal.lastRedraw()>=MDH_FRAME_TIME) ) {
			terminal.redraw(t2);
		}
	}

	/** Handle Enter on program entry line. */
	commandEnter() {
		let x=terminal.cursor()[0], y=terminal.cursor()[1];
		let line=terminal.textRange([0,y], [terminal.termsize()[0]-1,y]).trim();
		terminal.print("\n");
		DLOG("command: "+line);
		try {
			if (line) {
				this.parse(line);
			}
		} catch (e) {
			console.error(e);
			if (e instanceof BasicError) {
				terminal.fgcolor=TermColor.red;
				terminal.print(e.message+"\n");
				terminal.fgcolor=TermColor.white;
			} else {
				terminal.fgcolor=TermColor.red;
				terminal.print("ERROR: "+e.message+"\n");
				terminal.fgcolor=TermColor.white;
			}
		}
	}

	breakKey() {
		terminal.fgcolor=15; terminal.bgcolor=0;
		terminal.print("<BREAK>\n");
		this.state=TBW_STATE_CMD;
	}

	/** Handle Enter on input entry line, call inputCallback(line). */
	inputEnter() {
		let y=terminal.cursor()[1];
		let line=terminal.textRange([this.inputLeft,y], [terminal.termsize()[0]-1,y]).trim();
		terminal.print("\n");
		DLOG("inputEnter "+line);
		this.inputCallback(line);
	}

	runtimeError(msg) {
		console.error(msg);
		terminal.fgcolor=TermColor.red;
		terminal.print(msg+"\n");
		terminal.fgcolor=TermColor.white;
		this.doEnd();
	}

	setInkey(key) {
		let c=key;
		switch (key) {
			case "Enter":		c="\n"; break;
			case "Backspace":	c="\b"; break;
			case "Tab":			c="\t"; break;
			case "Escape":		c="\x1b"; break;
			case "Delete":		c="\x7f"; break;
			case "Help":
			case "Insert":		c="\x1a"; break;
			case "Home":		c="\x1c"; break;
			case "End":			c="\x1d"; break;
			case "PageUp":		c="\x1e"; break;
			case "PageDown":	c="\x1f"; break;
			case "ArrowUp":		c=TermArrows[0]; break;
			case "ArrowRight":	c=TermArrows[1]; break;
			case "ArrowDown":	c=TermArrows[2]; break;
			case "ArrowLeft":	c=TermArrows[3]; break;
		}
		this.inkey=c;
	}

	parseError(p, msg) {
		let e="SYNTAX: "+msg+": ";
		e += p.line.substring(0, Math.max(0,p.i))+"^"+p.line.substring(Math.max(0,p.i));
		e += ": c="+p.c+", tok="+p.tok+", tokens="+p.tokens.join(":");
		throw new BasicError(e);
	}

	/** advance cursor, set p.c, return true, or false if at EOL. */
	parseNextChar(p) {
		if (p.i>=-1 && p.i<p.line.length-1) {
			p.c=p.line[++p.i];
			//DLOG("nextchar ["+p.i+"]="+p.c);
			return true;
		} else {
			p.c='';
			p.i=p.line.length;
			return false;
		}
	}

	/** reverse cursor to last occurence of `c`. */
	parsePushbackChar(p, c) {
		while (p.i>=0 && p.line[p.i] !== c) {
			//DLOG("    parsePushbackChar("+c+"): "+ p.line.substring(0, Math.max(0,p.i))+"^"+p.line.substring(Math.max(0,p.i)) );
			--p.i;
		}
		//DLOG("AFTER parsePushbackChar("+c+"): "+ p.line.substring(0, Math.max(0,p.i))+"^"+p.line.substring(Math.max(0,p.i)) );
	}

	/** Advance cursor as long as p.c is whitespace. */
	parseWhitespace(p) {
		while (isWhitespace(p.c)) {
			this.parseNextChar(p);
		}
	}

	parseEOL(p) {
		this.parseWhitespace(p);
		if (this.parseNextChar(p)) {
			this.parseError(p, "junk after "+p.tokens.join(" "));
		}
	}

	parseLabel(p) {
		//DLOG("parseLabel "+JSON.stringify(p));
		p.tok += p.c;
		// TODO: name label
		while (this.parseNextChar(p) && isDigit(p.c)) {
			p.tok += p.c;
		}
		this.parseWhitespace(p);
		let n=parseInt(p.tok,10);
		//DLOG("    tok="+p.tok+", n="+n+", tokens="+p.tokens+", line="+p.line.substring(p.i));
		if (n>=1 && n<=TBW_LINENUM_MAX) {
			p.tokens.push( new BasicLabel(null, n) );
			p.tok="";
		} else {
			this.parseError(p, "bad line number");
		}
	}

	parseNumber(p) {
		//DLOG("parseNumber: "+JSON.stringify(p));
		do {
			if (isDigit(p.c) ||
				(p.c==='-' && (p.tok==="" || p.tok.endsWith('E'))) ||
				(p.c==='.' && p.tok.indexOf('.')<0) ||
				((p.c==='e' || p.c==='E')  && p.tok.indexOf('E')<0) ) {
				p.tok += p.c.toUpperCase();
			} else {
				let n=parseFloat(p.tok);
				if ( ! Number.isNaN(n)) {
					p.tokens.push( new BasicNum(n) );
					//DLOG("    parseNumber got "+n);
					p.tok="";
					break;
				} else {
					this.parseError(p, "invalid number");
				}
			}
		} while (this.parseNextChar(p));
		this.parseWhitespace(p);
		//DLOG("END parseNumber: "+JSON.stringify(p));
	}

	parseString(p) {
		//DLOG("parseString: "+JSON.stringify(p));
		// don't store first quote
		while (this.parseNextChar(p)) {
			if (p.c==='"') {
				p.tokens.push( new BasicString(p.tok) );
				//DLOG("    parseString got "+p.tok);
				p.tok="";
				this.parseNextChar(p);
				break;
			} else {
				p.tok += p.c;
			}
		}
		if (p.tok.length>0) {
			this.parseError(p, "unterminated string");
		}
		this.parseWhitespace(p);
		//DLOG("end of parseString: "+JSON.stringify(p));
	}

	parseVarname(p) {
		//DLOG("parseVarname: "+JSON.stringify(p));
		let isString=false;
		while ( isAlpha(p.c) || (isDigit(p.c) && p.tok.length>=1) ||
				(p.c==='$' && p.tok.length>=1) ) {
			if (p.c==='$') {
				isString=true;
			}
			p.tok+=p.c.toUpperCase();
			this.parseNextChar(p);
			if (isString) { break; }
		}
		if (TBW_FUNCTIONS[p.tok] || p.tok.length<=(isString?3:2)) { // TODO: level 1
			// OK
		} else {
			this.parseError(p, "variable name too long");
		}
		let vartok=new BasicVar(p.tok, null);
		p.tok="";
		p.tokens.push(vartok);

		// parse paren, exprs
		if (p.c==='(') {
			//DLOG("    BEGIN arguments "+JSON.stringify(p));
			let i=p.tokens.length; // collect tokens after this
			this.parseNextChar(p);
			for (;;) {
				if (p.c===',') {
					this.parseNextChar(p);
					this.parseWhitespace(p);
					// next
				} else if (p.c===')') {
					this.parseNextChar(p);
					break; // end
				} else {
					this.parseTerm(p);
				}
			}
			vartok.argList=p.tokens.splice(i, p.tokens.length);
			//DLOG("    END arguments="+JSON.stringify(vartok.argList));
		}

		this.parseWhitespace(p);
	}

	parseWord(p, word) {
		//DLOG("parseWord: "+JSON.stringify(p)+", "+word);
		while (isAlpha(p.c)) {
			p.tok += p.c.toUpperCase();
			this.parseNextChar(p);
		}
		if (p.tok===word) { // TODO: level 1
			// OK
		} else {
			this.parseError(p, "expected word '"+word+"'");
		}
		p.tokens.push( new BasicStmt(p.tok) );
		p.tok="";
		this.parseWhitespace(p);
	}

	parseOp(p) {
		//DLOG("parseOp: "+JSON.stringify(p));
		if ("+-*/%^=<>".indexOf(p.c)>=0) {
			p.tok=p.c;
			this.parseNextChar(p);
			if ( (p.tok==="<" && (p.c===">" || p.c==="=")) ||
					(p.tok===">" && p.c==="=") ) {
				p.tok += p.c;
				this.parseNextChar(p);
			}
			let op=p.tok;
			p.tok="";
			//DLOG("    parseOp got "+op);
			p.tokens.push( new BasicOp(op) );
			this.parseWhitespace(p);
			return op;
		} else {
			DLOG("    no op");
			return false;
		}
	}

	parseTerm(p) {
		if (DEBUG) {
			// prevent infinite loops
			if (--termLimit<=0) {
				throw new BasicError("Too many terms.");
			}
		}

		//DLOG("BEGIN parseTerm: "+JSON.stringify(p));
		if (isDigit(p.c) || p.c==='-' || p.c==='.') {
			this.parseNumber(p);
		} else if (p.c==='"') {
			this.parseString(p);
		} else if (isAlpha(p.c)) {
			this.parseVarname(p);
		} else if (p.c==='(') {	// TODO: should store BasicExpr
			this.parseExpr(p);
		} else {
			this.parseError(p, "unexpected term");
		}
		this.parseWhitespace(p);
		//DLOG("END parseTerm: "+JSON.stringify(p));
	}

	parseExpr(p) {
		//DLOG("BEGIN parseExpr: "+JSON.stringify(p));
		if (p.c==='(') {
			let i=p.tokens.length; // collect tokens after this
			if (this.parseNextChar(p)) {
				this.parseTerm(p);
				while (this.parseOp(p)) {
					this.parseTerm(p);
				}

				if (p.i<p.line.length && p.c===')') {
					this.parseNextChar(p);
					let argtoks=p.tokens.splice(i, p.tokens.length);
					p.tokens.push( new BasicExpr(argtoks) );
					//DLOG("END parseExpr: "+JSON.stringify(p));
				} else {
					this.parseError(p, "expected close paren");
				}
			} else {
				this.parseError(p, "end before close paren");
			}
		} else {
			this.parseError(p, "expected open paren");
		}
	}

	parseStmtDimension(p) {
		this.parseVarname(p);
		const varname=p.tokens[p.tokens.length-1];
		if (varname.argList===null||varname.argList.length===0) {
			this.parseError(p, "array varname needs 1+ dimensions");
		}
		this.parseEOL(p);
	}

	parseStmtDir(p) {
		if (p.i<p.line.length) {
			this.parseTerm(p);
		}
		this.parseEOL(p);
	}

	parseStmtGoto(p) {
		this.parseLabel(p);
		this.parseEOL(p);
	}

	parseStmtIf(p) {
		this.parseTerm(p);
		this.parseWord(p, "THEN");
		this.parseStatement(p);
	}

	parseStmtInput(p) {
		if (p.c==='"') {
			this.parseString(p);
			if (p.c===',') {
				p.tokens.push( new BasicSep(p.c) );
				this.parseNextChar(p);
				this.parseWhitespace(p);
			} else {
				this.parseError(p, "comma expected");
			}
		}
		for (;;) {
			this.parseVarname(p);
			if (p.c===',') {
				p.tokens.push( new BasicSep(p.c) );
				this.parseNextChar(p);
				this.parseWhitespace(p);
			} else {
				break;
			}
		}
		this.parseEOL(p);
	}

	parseStmtIncDec(p) {
		this.parseVarname(p);
		this.parseEOL(p);
	}

	parseStmtLet(p) {
		this.parseVarname(p);
		if (p.c==='=') {
			p.tokens.push( new BasicOp('=') );
			if (this.parseNextChar(p)) {
				this.parseWhitespace(p);
				this.parseTerm(p);
			} else {
				this.parseError(p, "term expected");
			}
		} else {
			this.parseError(p, "= expected");
		}
		this.parseEOL(p);
	}

	parseStmtList(p) {
		let lineCount=0;
		if (p.i<p.line.length) {
			this.parseLabel(p);
		}
		if (p.c===',') {
			p.tokens.push( new BasicSep(p.c) );
			this.parseNextChar(p);
			this.parseWhitespace(p);
		}
		if (p.i<p.line.length) {
			this.parseLabel(p);
		}
		this.parseEOL(p);
	}

	parseStmtLoad(p) {
		this.parseTerm(p);
		this.parseEOL(p);
	}

	parseStmtPause(p) {
		this.parseTerm(p);
		this.parseEOL(p);
	}

	parseStmtPrint(p) {
		let lastComma=false;
		do {
			if (p.c==='') {
				break;
			} else if (p.c===',' || p.c===';') {
				if ( ! lastComma) {
					p.tokens.push( new BasicSep(p.c) );
					lastComma=true;
					this.parseNextChar(p);
					this.parseWhitespace(p);
					//DLOG("comma: "+JSON.stringify(p));
				} else {
					this.parseError(p, "comma repeated");
				}
			} else {
				lastComma=false;
				this.parseTerm(p);
			}
		} while (p.i<p.line.length);
		this.parseEOL(p);
	}

	parseStmtRedir(p) {
		this.parseTerm(p);
		this.parseEOL(p);
	}

	parseStmtRem(p) {
		let s=p.line.substr(p.i).trim();
		// NOTE: if I add string escapes, escape remark strings
		p.tokens.push( new BasicString(s.replaceAll('"',"'")) );
		p.i=p.line.length;
	}

	parseStmtRun(p) {
		if (p.i<p.line.length) {
			this.parseTerm(p);
		}
		this.parseEOL(p);
	}

	parseStatement(p) {
		p.tok += p.c;
		if (p.c==="'" || p.c==="?") {
			this.parseNextChar(p);
		} else {
			while (this.parseNextChar(p) && isAlpha(p.c)) {
				p.tok += p.c;
			}
		}
		let cmd=p.tok.toUpperCase();
		//DLOG("    tok="+p.tok+", cmd="+cmd+", tokens="+p.tokens+", line="+p.line.substring(p.i));
		p.tok="";
		this.parseWhitespace(p);
		switch (cmd) {
			case "CLS":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseEOL(p);
				break;
			case "DIM":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtDimension(p);
				break;
			case "DIR":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtDir(p);
				break;
			case "END":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseEOL(p);
				break;
			case "GOTO": case "GOSUB":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtGoto(p);
				break;
			case "IF":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtIf(p);
				break;
			case "INC":
			case "DEC":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtIncDec(p);
				break;
			case "INPUT":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtInput(p);
				break;
			case "LET":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtLet(p);
				break;
			case "LIST":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtList(p);
				break;
			case "LLIST":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtList(p);
				break;
			case "LOAD":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtLoad(p);
				break;
			case "LPRINT":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtPrint(p);
				break;
			case "NEW":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseEOL(p);
				break;
			case "PAUSE":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtPause(p);
				break;
			case '?': case "PRINT":
				p.tokens.push( new BasicStmt("PRINT") );
				this.parseStmtPrint(p);
				break;
			case "REDIR":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtRedir(p);
				break;
			case "'": case "REM":
				p.tokens.push( new BasicStmt("REM") );
				this.parseStmtRem(p);
				break;
			case "RETURN":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseEOL(p);
				break;
			case "RUN":
				p.tokens.push( new BasicStmt(cmd) );
				this.parseStmtRun(p);
				break;
			// TODO: more statements
			default:
				this.parseError(p, "unknown statement");
		}
	}

	parse(linein) {
		if (linein.length===0) {
			return;
		}
		let p={	// parser state
			line: linein+" ",
			i: -1,
			c: '',
			tok: "",
			tokens: [],
		};
		if (this.parseNextChar(p)) {
			if (isDigit(p.c)) {
				this.parseLabel(p);
				this.parseStatement(p);
			} else if (isAlpha(p.c) || p.c==="'" || p.c==="?") {
				this.parseStatement(p);
			} else {
				this.parseError(p, "invalid token");
			}
			DLOG("parsed: "+p.tokens.join(":")+JSON.stringify(p));
			// for (let i=0; i<p.tokens.length; ++i) {
			// 	DLOG("    ["+i+"]: ("+typeof(p.tokens[i])+")"+p.tokens[i]);
			// }
			// if (DEBUG) {
			// 	terminal.fgcolor=TermColor.gray;
			// 	terminal.print(p.tokens.join(":")+"\n");
			// 	terminal.fgcolor=TermColor.white;
			// }

			if (p.tokens.length===0) {
				// nothing
			} else if (p.tokens[0] instanceof BasicLabel) {
				this.programAdd(p.tokens);
			} else {
				this.pc=-1;
				this.eval(p.tokens, 0);
			}
		}
	}

	programAdd(tokens) {
		let linenum=tokens[0].num;
		this.program[linenum]=tokens.slice(1);
		this.programStart=Math.min(this.programStart, linenum);
		this.programEnd=Math.max(this.programEnd, linenum);
	}

	labelGet(lbl) {
		if (lbl instanceof BasicLabel) {
			if (lbl.num) {
				return lbl.num;
			}
			// TODO: L2 named labels
		} else if (isNumber(lbl) && lbl>=1 && lbl<=TBW_LINENUM_MAX) {
			return lbl;
		} else {
			throw new BasicError("Invalid label "+lbl+" at line "+this.pc);
		}
	}

	varGet(varref) {
		//DLOG("varGet "+varref);
		const f=TBW_FUNCTIONS[varref.name];
		if (f) {
			const args=[];
			if (varref.argList) {
				for (let a of varref.argList) {
					args.push( this.evalExpr(a) );
				}
			}
			return f(args);
		}
		const varval=this.variables[varref.name];
		if (varval===undefined) {
			if (varref.argList!==null) {
				throw new BasicError("Get from undefined array "+varref.toList()+" at line "+this.pc);
			}
			return varref.isString() ? "" : 0;
		} else {
			if (isArray(varval)) {
				if (varref.argList===null) {
					throw new BasicError("Get from base name "+varref.toList()+" of array at line "+this.pc);
				}
				const dims=[];
				for (let a of varref.argList) {
					dims.push( this.evalExpr(a) );
				}
				try {
					return arrayGetMulti(varval, dims);
				} catch (e) {
					throw new BasicError(e.toString()+" at line "+this.pc);
				}
			} else {
				return varval;
			}
		}
	}

	varSet(varref, value) {
		//DLOG("varSet "+varref+"="+value);
		if (TBW_FUNCTIONS[varref.name]) {
			throw new BasicError("Set to reserved word "+varref.name+" at line "+this.pc);
		}
		if (varref.isString()) {
			if (isString(value)) {
				if (value.length>=TBW_STRING_MAX) {
					value=substring(0, TBW_STRING_MAX);
				}
			} else {
				value=value.toString();
			}
		} else { // numeric var
			if (isString(value)) {
				let n=parseFloat(value,10);
				value=isNumber(n) ? n : 0;
			}
		}

		let varval=this.variables[varref.name];
		if (isArray(varval)) {
			if (varref.argList===null) {
				throw new BasicError("Set to base name "+varref.toList()+" of array at line "+this.pc);
			}
			const dims=[];
			for (let a of varref.argList) {
				dims.push( this.evalExpr(a) );
			}
			try {
				arraySetMulti(varval, dims, value);
			} catch (e) {
				throw new BasicError(e.toString()+" at line "+this.pc);
			}
		} else {
			if (varref.argList!==null) {
				throw new BasicError("Set to undefined array "+varref.toList()+" at line "+this.pc);
			}
			this.variables[varref.name]=value;
		}

		//DLOG("variables="+JSON.stringify(this.variables));
	}

	assertFilename(x) {
		if (isString(x) && x.length<=16 && x.match(TBW_FILENAME_PATTERN)) {
			return x.toLowerCase();
		} else {
			throw new BasicError("Expected filename but got "+x+" at line "+this.pc);
		}
	}

	assertNumber(x) {
		if (isNumber(x)) {
			return x;
		} else {
			throw new BasicError("Expected number but got "+x+" at line "+this.pc);
		}
	}

	assertString(x) {
		if (isString(x)) {
			return x;
		} else {
			throw new BasicError("Expected string but got "+x+" at line "+this.pc);
		}
	}

	/** Returns 1 for true, 0 for false. */
	boolNumber(x) {
		return x ? 1 : 0;
	}

	/** Returns false for 0 or "", true for all other values. */
	boolForValue(x) {
		return (isNumber(x) && x !== 0) || (isString(x) && x !== "");
	}

	/** Returns boolean 0/1 result of op(a, b) if they are of the same type. */
	evalCompare(x, y, cmp) {
		if (isNumber(x)) {
			return this.boolNumber( cmp(this.assertNumber(x), this.assertNumber(y)) );
		} else {
			return this.boolNumber( cmp(this.assertString(x).toUpperCase(), this.assertString(y).toUpperCase()) );
		}
	}

	/** Returns numeric result of op(a, b) if they are both numbers. */
	evalMath(x, y, op) {
		return op(this.assertNumber(x), this.assertNumber(y));
	}

	/** Evaluates a single token, returns value. */
	// TODO: fix every use of rc[0], rc[1] (next index, don't need)
	evalExpr(tok) {
		// if (DEBUG) {
		// 	// prevent infinite loops
		// 	if (--termLimit<=0) {
		// 		throw new BasicError("Too many terms.");
		// 	}
		// }

		//DLOG("BEGIN evalExpr "+tok);
		if (tok instanceof BasicNum) {
			//DLOG("    number="+tok.num);
			return tok.num;
		} else if (tok instanceof BasicString) {
			//DLOG("    string="+tok.name);
			return tok.name;
		} else if (tok instanceof BasicVar) {
			// TODO: array or function
			const value=this.varGet(tok);
			//DLOG("    var "+tok.name+"="+value);
			return value;
		} else if (tok instanceof BasicExpr) {
			//DLOG("    BEGIN expr "+tok);
			if (tok.argList.length===0) {
				return 0;
			}
			let value=this.evalExpr(tok.argList[0]);
			for (let i=1; i<tok.argList.length; i+=2) {
				//DLOG("    i="+i+", value="+value);
				let op=tok.argList[i];
				if (op instanceof BasicOp) {
					let next=this.evalExpr(tok.argList[i+1]);
					//DLOG("    DO i="+i+", expr "+value+" "+op+" "+next);
					switch (op.name) {
						case '+':
							if (isString(value)) {
								value=value+next.toString();
							} else {
								value=this.evalMath(value, next, (x,y)=>{return x+y});
							}
							break;
						case '-':
							value=this.evalMath(value, next, (x,y)=>{return x-y});
							break;
						case '*':
							value=this.evalMath(value, next, (x,y)=>{return x*y});
							break;
						case '/': {
							value=this.evalMath(value, next, (x,y)=>{
								if (y !== 0) {
									return x/y;
								} else {
									throw new BasicErrro("Divide by 0 at line "+this.pc);
								}
							});
							break;
						}
						case '%': {
							value=this.evalMath(value, next, (x,y)=>{
								x=Math.floor(x);
								y=Math.floor(y);
								if (y !== 0) {
									return x%y;
								} else {
									throw new BasicErrro("Modulo by 0 at line "+this.pc);
								}
							});
							break;
						}
						case '^':
							value=this.evalMath(value, next, (x,y)=>{return x**y});
							break;
						case '=':
							value=this.evalCompare(value, next, (x,y)=>{return x==y});
							break;
						case '<>':
							value=this.evalCompare(value, next, (x,y)=>{return x!=y});
							break;
						case '<':
							value=this.evalCompare(value, next, (x,y)=>{return x<y});
							break;
						case '<=':
							value=this.evalCompare(value, next, (x,y)=>{return x<=y});
							break;
						case '>':
							value=this.evalCompare(value, next, (x,y)=>{return x>y});
							break;
						case '>=':
							value=this.evalCompare(value, next, (x,y)=>{return x>=y});
							break;
						default:
							throw new BasicError("INVALID op "+op+" at line "+this.pc);
					}
				} else {
					throw new BasicError("Expected binary op but got "+tok+" at line "+this.pc);
				}
			}

			//DLOG("    END expr parens "+tok+": "+value);
			return value;
		} else {
			throw new BasicError("INVALID type "+tok+" at line "+this.pc);
		}
	}

	doCls() {
		terminal.cls();
		this.nextPC();
	}

	doDimension(tokens, toki) {
		for (let i=toki+1; i<tokens.length; i+=2) {
			const tok=tokens[i];
			//DLOG("    dimension "+tok);
			if (tok instanceof BasicVar) {
				const varref=tok;
				if (this.variables[varref.name]!==undefined) {
					throw new BasicError("Redimension of variable "+varref.toList()+" at line "+this.pc);
				}
				let dims=[], dimTotal=1;
				for (let j=0; j<varref.argList.length; ++j) {
					let d=this.evalExpr(varref.argList[j]);
					if (isNumber(d) && d>=1) {
						++d; // DIM A(10) creates 0..10
						dims.push(d);
						dimTotal*=d;
					} else {
						throw new BasicError("INVALID dimension "+d+" at line "+this.pc);
					}
				}
				if (dimTotal>=TBW_DIM_MAX) {
					throw new BasicError("INVALID dimensions cannot exceed "+TBW_DIM_MAX+" at line "+this.pc);
				}
				this.variables[varref.name]=arrayDimMulti(dims, varref.isString() ? "" : 0);
			}
		}
		this.nextPC();
	}

	doDir(tokens, toki) {
		let filename;
		if (toki+1<tokens.length) {
			let rc=this.evalExpr(tokens[toki+1]);
			filename=this.assertString(rc);
		} else {
			filename="";
		}
		if (filename==="") {
			// OK
		} else {
			this.assertFilename(filename);
		}
		DLOG("doDir "+filename);
		this.state=TBW_STATE_WAIT;
		ajax("GET", "disk/", {}, (data,err)=>{ this.dirResult(data,err,filename) });
	}

	dirResult(data, err, filename) {
		if (err) {
			throw new BasicError("DIR error: "+err+" at line "+this.pc);
		}
		if (data) {
			let dataLines=strSplitLines(data);
			let filenames=[];
			for (let line of dataLines) {
				let m=line.match(TBW_DIR_PATTERN);
				if (m && m[1].indexOf(filename)>=0) {
					filenames.push(m[1]);
			}}

			let w=20, ncols=Math.floor(terminal.termsize()[0] / w);
			let nrows=Math.ceil(filenames.length / ncols);
			DLOG("lines="+JSON.stringify(filenames)+" ncols="+ncols+", nrows="+nrows);
			let s="";
			for (let y=0; y<nrows; ++y) {
				for (let x=0; x<ncols; ++x) {
					let i=x * nrows + y;
					if (i<filenames.length) {
						s += strPadRight(filenames[i], w);
					} else {
						s += strPadRight("", w);
					}
				}
				s += "\n";
			}
			terminal.print(s);
		}
		if (this.pc>=1) {
			this.state=TBW_STATE_RUN;
			this.inkey=null;
			this.nextPC();
		} else {
			this.readyPrompt();
		}
	}

	doEnd() {
		this.pc=0;
		this.readyPrompt();
	}

	doGosub(tokens, toki) {
		this.substack.push(this.pc+1);
		this.pc=this.labelGet(tokens[toki+1]);
		// if command, switch to run mode
	}

	doGoto(tokens, toki) {
		this.pc=this.labelGet(tokens[toki+1]);
		// if command, switch to run mode
	}

	doIf(tokens, toki) {
		// IF, term, THEN, stmt
		let oldpc=this.pc;
		let rc=this.evalExpr(tokens[toki+1]);
		if (this.boolForValue(rc)) {
			this.eval(tokens, toki+3); // toki+2=THEN
		} else {
		}
		// move PC only if not GOTO or multiple IFs
		if (toki<=1 && this.pc===oldpc) {
			this.nextPC();
		}
	}

	doInput(tokens, toki) {
		//INPUT ["PROMPT" ,] varname
		if (tokens[toki+1] instanceof BasicString) {
			terminal.print(tokens[toki+1].name);
			toki += 3;
		} else {
			terminal.print("? ");
			toki++;
		}
		this.state=TBW_STATE_INPUT;
		this.inputLeft=terminal.cursor()[0];
		this.inputCallback=(line)=>{
			this.varSet(tokens[toki], line);
			this.nextPC();
			this.state=TBW_STATE_RUN;
			this.inkey=null;
		};
	}

	doIncDec(tokens, toki) {
		// LET, varname, =, X
		const delta=tokens[toki].name==="INC" ? 1 : -1;
		const varref=tokens[toki+1];
		let value=this.varGet(varref);
		if (isString(value)) {
			let n=parseFloat(value,10);
			value=isNumber(n) ? n : 0;
		}
		this.varSet(varref, value+delta);
		this.nextPC();
	}

	doLet(tokens, toki) {
		// LET, varname, =, X
		const varref=tokens[toki+1];
		let rc=this.evalExpr(tokens[toki+3]);
		this.varSet(varref, rc);
		// TODO: L2 comma
		this.nextPC();
	}

	doList(tokens, toki) {
		// LIST, a, SEP, b
		let lpr=tokens[toki].name==="LLIST";
		let aline=1, bline=TBW_LINENUM_MAX, dig=1;
		if (toki+1<tokens.length) {
			aline=this.labelGet(tokens[toki+1]);
		}
		if (toki+3<tokens.length) {
			bline=this.labelGet(tokens[toki+3]);
		}
		if (aline>bline) {
			const tmp=aline;
			aline=bline;
			bline=tmp;
		}
		aline=Math.max(aline, this.programStart);
		bline=Math.min(bline, this.programEnd);
		if (aline>=10000 || bline>=10000) {
			dig=5;
		} else if (aline>=1000 || bline>=1000) {
			dig=4;
		} else if (aline>=100 || bline>=100) {
			dig=3;
		} else if (aline>=10 || bline>=10) {
			dig=2;
		}
		//DLOG("    LIST "+aline+","+bline);
		for (let i=aline; i<=bline; ++i) {
			let tokens=this.program[i];
			if (tokens) {
				let s=strPadLeft(""+i,dig,"0");
				for (let j=0; j<tokens.length; ++j) {
					s += tokens[j].toList();
				}
				s += "\n";
				if (lpr) {
					terminal.output(s);
				} else {
					terminal.print(s);
				}
			}
		}
		this.nextPC();
	}

	doLoad(tokens, toki, runAfter) {
		let rc=this.evalExpr(tokens[toki+1]);
		let filename=this.assertFilename(rc);
		DLOG("doLoad "+filename+", "+runAfter);
		this.doNew();
		this.state=TBW_STATE_WAIT;
		ajax("GET", "disk/"+filename, {}, (data,err)=>{ this.loadResult(data,err,runAfter) });
	}

	loadResult(data, err, runAfter) {
		if (err) {
			console.error(err);
			let msg=arrayFirst(strSplitLines(err));
			this.runtimeError("LOAD error: "+msg+" at line "+this.pc);
			return;
		}
		if (data) {
			for (let line of strSplitLines(data)) {
				this.parse(line);
			}
		}
		if (runAfter) {
			this.pc=this.programStart;
			this.state=TBW_STATE_RUN;
			this.inkey=null;
		} else {
			this.readyPrompt();
		}
	}

	doNew() {
		this.program=[];
		this.variables={};
		this.substack=[];
		this.pc=-1;
		this.programStart=TBW_LINENUM_MAX;
		this.programEnd=1;
		this.inputCallback=null;
		this.state=TBW_STATE_CMD;
	}

	doPause(tokens, toki) {
		let rc=this.evalExpr(tokens[toki+1]);
		if (isNumber(rc)) {
			this.pauseUntil=Date.now()+rc*1000.0;
		} else {
			throw new BasicError("PAUSE takes numeric seconds at line "+this.pc);
		}
		this.nextPC();
	}

	doPrint(tokens, toki) {
		let lpr=tokens[toki].name==="LPRINT";
		let anything=false, lastComma=false, s="";
		for (let i=toki+1; i<tokens.length; ++i) {
			const tok=tokens[i];
			//DLOG("    print tokens["+i+"]="+tok);
			if (tok instanceof BasicSep) {
				if (tok.name===',') {
					s+="\t";
					anything=true;
					lastComma=true;
				} else if (tok.name===';') {
					// nothing
					lastComma=true;
				} else {
					throw new BasicError("Invalid separator "+tok);
				}
			} else {
				let rc=this.evalExpr(tokens[i]);
				s+=rc.toString();
				anything=true;
				lastComma=false;
			}
			if (s.length>0 && ! lpr) {
				terminal.print(s);
				s="";
			}
		}
		if ( ! anything) {
			s+=" ";
		}
		if ( ! lastComma) {
			s+="\n";
		}
		if (s.length>0) {
			if (lpr) {
				terminal.output(s);
			} else {
				terminal.print(s);
			}
		}
		this.nextPC();
	}

	doRedir(tokens, toki) {
		let rc=this.evalExpr(tokens[toki+1]);
		let url=rc;
		DLOG("doRedir "+url);
		this.state=TBW_STATE_WAIT;
		if (this.timer) {
			this.timer.stop();
		}
		location=url;
	}

	doReturn() {
		if (this.substack.length>0) {
			this.pc=this.substack.pop();
		} else {
			throw new BasicError("RETURN without GOSUB at line "+this.pc);
		}
	}

	doRun(tokens, toki) {
		if (DEBUG) { termLimit=2000; }
		this.variables={};
		if (toki+1<tokens.length) {
			this.doLoad(tokens, toki, true);
		} else if (this.program.length>0) {
			this.pc=this.programStart;
			this.state=TBW_STATE_RUN;
			this.inkey=null;
		} else {
			this.doEnd();
		}
	}

	nextPC() {
		if (this.pc<=0) {
			return;
		}
		do {
			this.pc++;	// FIXME: smarter next line
		} while (this.pc<=this.programEnd && ! this.program[this.pc]);
	}

	eval(tokens, toki) {
		//DLOG("eval: "+this.pc+" "+tokens.join(":")+"["+toki+"]");
		try {
			if (tokens[toki] instanceof BasicStmt) {
				switch (tokens[toki].name) {
					case "CLS":
						this.doCls();
						break;
					case "DIM":
						this.doDimension(tokens, toki);
						break;
					case "DIR":
						this.doDir(tokens, toki);
						break;
					case "END":
						this.doEnd();
						break;
					case "GOTO":
						this.doGoto(tokens, toki);
						break;
					case "GOSUB":
						this.doGosub(tokens, toki);
						break;
					case "IF":
						this.doIf(tokens, toki);
						break;
					case "INC":
					case "DEC":
						this.doIncDec(tokens, toki)
						break;
					case "INPUT":
						this.doInput(tokens, toki);
						break;
					case "LET":
						this.doLet(tokens, toki);
						break;
					case "LIST": case "LLIST":
						this.doList(tokens, toki);
						break;
					case "LOAD":
						this.doLoad(tokens, toki, false);
						break;
					case "NEW":
						this.doNew();
						break;
					case "PAUSE":
						this.doPause(tokens, toki);
						break;
					case '?': case "PRINT": case "LPRINT":
						this.doPrint(tokens, toki);
						break;
					case "REDIR":
						this.doRedir(tokens, toki);
						break;
					case "'": case "REM":
						this.nextPC();
						break;
					case "RETURN":
						this.doReturn();
						break;
					case "RUN":
						this.doRun(tokens, toki);
						break;
					// TODO: more statements
					default:
						throw new BasicError("Invalid immediate statement: "+tokens.join(" ")+"["+toki+"]");
				}
			} else {
				throw new BasicError("Expected statement, but got: "+tokens.join(" ")+"["+toki+"]");
			}
		} catch (e) {
		    this.runtimeError(e.message);
		} finally {
		}
	}

} // TinyBasicWeb

var termLimit=2000; // FIXME: debug to prevent infinite loops

let terminal=null;
let tbw=null;

function tbasicwebinit(termid, termsize, charsize, filename) {
	terminal=new Terminal(termid, termsize, charsize);
	tbw=new TinyBasicWeb();
	if (filename) {
		tbw.parse("RUN \""+filename+"\"");
	}
}

