/* terminal.js
 * Cyberhole
 * Created 2025-07-11
 * Copyright © 2025 by Mark Damon Hughes. All Rights Reserved.
 * See BSDLicense.txt
 */

// requires mdhutil.js
// TODO: copy, paste
// TODO: pixel chars, line drawing commands

/* eslint-env browser, es2020 */

"use strict";

//---------------------------------------
// Terminal

/** Arrows up, right, down, left */
const TermArrows=[
	"\u2191",	// 0 = ↑
	"\u2192",	// 1 = →
	"\u2193",	// 2 = ↓
	"\u2190",	// 3 = ←
];

/** 2x2 bitmap chars with lit quadrants 1 2 / 4 8: */
const TermPixels=[
	" ",		// 00 =
	"\u2598",	// 01 = ▘
	"\u259d",	// 02 = ▝
	"\u2580",	// 03 = ▀
	"\u2596",	// 04 = ▖
	"\u258c",	// 05 = ▌
	"\u259e",	// 06 = ▞
	"\u259b",	// 07 = ▛
	"\u2597",	// 08 = ▗
	"\u259a",	// 09 = ▚
	"\u2590",	// 10 = ▐
	"\u259c",	// 11 = ▜
	"\u2584",	// 12 = ▄
	"\u2599",	// 13 = ▙
	"\u259f",	// 14 = ▟
	"\u2588",	// 15 = █
];

/** All of the thin line chars, counting connections:
 * . 1 .
 * 8 + 2
 * . 4 .
 *
 * Lines are drawn to connections, except the side-only pieces cover the whole side. Thus:
 * 1 is:	2 is:		3 is:		etc.
 * [*****]	[....*]		[..*..]
 * [.....]	[....*]		[..*..]
 * [.....]	[....*]		[..***]
 * [.....]	[....*]		[.....]
 * [.....]	[....*]		[.....]
 */
const TermLineChars=[
	" ",			// 00 =
	"\u2579",		// 01 = ╹
	"\u257a",		// 02 = ╺
	"\u2517",		// 03 = ┗
	"\u257B",		// 04 = ╻
	"\u2503",		// 05 = ┃
	"\u250F",		// 06 = ┏
	"\u2523",		// 07 = ┣
	"\u2578",		// 08 = ╸
	"\u251B",		// 09 = ┛
	"\u2501",		// 10 = ━
	"\u253B",		// 11 = ┻
	"\u2513",		// 12 = ┓
	"\u252B",		// 13 = ┫
	"\u2533",		// 14 = ┳
	"\u254B",		// 15 = ╋
];

const TermColor={
	black:		0,
	maroon:		1,
	green:		2,
	olive:		3,
	navy:		4,
	purple:		5,
	teal:		6,
	gray:		7,
	silver:		8,
	red:		9,
	lime:		10,
	yellow:		11,
	blue:		12,
	magenta:	13,
	cyan:		14,
	white:		15,
	graydark:	16,
	brown:		17,
	orange:		18,

	// compatibility hack
	aqua:		14,
};

const TERMCOLORMAX=18;

const TERMCOLORKEYS=[
	"black",	// 0
	"maroon",	// 1
	"green",	// 2
	"olive",	// 3
	"navy",		// 4
	"purple",	// 5
	"teal",		// 6
	"gray",		// 7
	"silver",	// 8
	"red",		// 9
	"lime",		// 10
	"yellow",	// 11
	"blue",		// 12
	"magenta",	// 13
	"cyan",		// 14
	"white",	// 15
	"graydark",	// 16
	"brown",	// 17
	"orange",	// 18
];

const TERMCOLORS=[
	"#000000",	//  0 = black
	"#800000",	//  1 = maroon
	"#008000",	//  2 = green
	"#808000",	//  3 = olive
	"#000080",	//  4 = navy
	"#800080",	//  5 = purple
	"#008080",	//  6 = teal
	"#808080",	//  7 = gray
	"#C0C0C0",	//  8 = silver
	"#FF0000",	//  9 = red
	"#00FF00",	// 10 = lime
	"#FFFF00",	// 11 = yellow
	"#0000FF",	// 12 = blue
	"#FF00FF",	// 13 = magenta
	"#00FFFF",	// 14 = cyan
	"#FFFFFF",	// 15 = white
	"#404040",	// 16 = greydark
	"#804000",	// 17 = brown
	"#FF8000",	// 18 = orange
	"rgba(0,0,0,0.50)",		// 19 = black50
	"rgba(255,255,0,0.50)",	// 20 = yellow50
];

const KEY_DEBOUNCE_TIME=100;
const TERMFLICKER=192;
const STATUS_TIME=2000;
const PIXEL_FONT_SIZE=8;

function termKeyPress(normkey, shkey=null, fnkey=null) {
	if ( ! terminal) {
		return false;
	}
	if ($(terminal.id+"-key-func").checked) {
		$(terminal.id+"-key-func").checked=false;
		if (fnkey !== null) {
			terminal.handleKey(fnkey);
		} else {
			terminal.beep();
		}
	} else if ($(terminal.id+"-key-shift").checked && shkey !== null) {
		terminal.handleKey(shkey);
	} else {
		terminal.handleKey(normkey);
	}
}

function termhtml(id, size, termsize) {
	return `<canvas id='${id}-canvas' class='terminal-canvas'
	width='${size[0]}' height='${size[1]}'></canvas>

<div id='${id}-key' class='terminal-key'>
<div>
	<span><button onclick='termKeyPress(String.fromCharCode(0x60), String.fromCharCode(0x7e))'>&#x60; ~</button></span>
	<span><button onclick='termKeyPress("1", "!", "F1")'>1 !</button></span>
	<span><button onclick='termKeyPress("2", "@", "F2")'>2 @</button></span>
	<span><button onclick='termKeyPress("3", "#", "F3")'>3 #</button></span>
	<span><button onclick='termKeyPress("4", "$", "F4")'>4 $</button></span>
	<span><button onclick='termKeyPress("5", "%", "F5")'>5 %</button></span>
	<span><button onclick='termKeyPress("6", "^", "F6")'>6 ^</button></span>
	<span><button onclick='termKeyPress("7", "&", "F7")'>7 &</button></span>
	<span><button onclick='termKeyPress("8", "*", "F8")'>8 *</button></span>
	<span><button onclick='termKeyPress("9", "(", "F9")'>9 (</button></span>
	<span><button onclick='termKeyPress("0", ")", "F10")'>0 )</button></span>
	<span><button onclick='termKeyPress("-", "_", "F11")'>- _</button></span>
	<span><button onclick='termKeyPress("+", "=", "F12")'>+ =</button></span>
</div>

<div style='margin-left:16px'>
	<span><button onclick='termKeyPress("Tab")'>TAB</button></span>
	<span><button onclick='termKeyPress("Q", "q")'>Q</button></span>
	<span><button onclick='termKeyPress("W", "w")'>W</button></span>
	<span><button onclick='termKeyPress("E", "e")'>E</button></span>
	<span><button onclick='termKeyPress("R", "r")'>R</button></span>
	<span><button onclick='termKeyPress("T", "t")'>T</button></span>
	<span><button onclick='termKeyPress("Y", "y")'>Y</button></span>
	<span><button onclick='termKeyPress("U", "u")'>U</button></span>
	<span><button onclick='termKeyPress("I", "i")'>I</button></span>
	<span><button onclick='termKeyPress("O", "o")'>O</button></span>
	<span><button onclick='termKeyPress("P", "p")'>P</button></span>
	<span><button onclick='termKeyPress(String.fromCharCode(0x5c), String.fromCharCode(0x7c))'>&#x5c; &#x7c;</button></span>
	<span><button onclick='termKeyPress("Backspace")' style='width:108px'>BS</button></span>
</div>

<div style='margin-left:32px'>
	<span><button onclick='termKeyPress("Escape")'>ESC</button></span>
	<span><button onclick='termKeyPress("A", "a")'>A</button></span>
	<span><button onclick='termKeyPress("S", "s")'>S</button></span>
	<span><button onclick='termKeyPress("D", "d")'>D</button></span>
	<span><button onclick='termKeyPress("F", "f")'>F</button></span>
	<span><button onclick='termKeyPress("G", "g")'>G</button></span>
	<span><button onclick='termKeyPress("H", "h")'>H</button></span>
	<span><button onclick='termKeyPress("J", "j")'>J</button></span>
	<span><button onclick='termKeyPress("K", "k")'>K</button></span>
	<span><button onclick='termKeyPress("L", "l")'>L</button></span>
	<span><button onclick='termKeyPress(";", ":")'>; :</button></span>
	<span><button onclick='termKeyPress(String.fromCharCode(0x22), String.fromCharCode(0x27))'>" '</button></span>
	<span><button onclick='termKeyPress("Enter")' style='width:108px'>CR</button></span>
</div>

<div>
	<span><button style='width:112px'><label for='${id}-key-shift'>⇑ <input type='checkbox' id='${id}-key-shift'></label></button></span>
	<span><button onclick='termKeyPress("Z", "z")'>Z</button></span>
	<span><button onclick='termKeyPress("X", "x")'>X</button></span>
	<span><button onclick='termKeyPress("C", "c")'>C</button></span>
	<span><button onclick='termKeyPress("V", "v")'>V</button></span>
	<span><button onclick='termKeyPress("B", "b")'>B</button></span>
	<span><button onclick='termKeyPress("N", "n")'>N</button></span>
	<span><button onclick='termKeyPress("M", "m")'>M</button></span>
	<span><button onclick='termKeyPress(",", String.fromCharCode(0x3c))'>, &lt;</button></span>
	<span><button onclick='termKeyPress(".", String.fromCharCode(0x3e))'>. &gt;</button></span>
	<span><button onclick='termKeyPress("?", "/")'>? /</button></span>
	<span><button onclick='termKeyPress("ArrowUp")'>↑</button></span>
</div>

<div>
	<span><button style='width:112px'><label for='${id}-key-func'>Fn <input type='checkbox' id='${id}-key-func'></label></button></span>
	<span><button onclick='termKeyPress(" ")' style='width:522px'>SPC</button></span>
	<span><button onclick='termKeyPress("ArrowLeft")'>←</button></span>
	<span><button onclick='termKeyPress("ArrowDown")'>↓</button></span>
	<span><button onclick='termKeyPress("ArrowRight")'>→</button></span>
</div>
</div>
<div>&nbsp; <button><label for='${id}-key-show'>Keyboard</label><input type='checkbox' id='${id}-key-show' checked></button>
</div>

<div id='${id}-output-wrapper' class='hidden'>
	<textarea id='${id}-output' class='terminal-output' cols='${termsize[0]}' rows='8'></textarea>
	<div>
		&nbsp; <button id='${id}-output-copy'>copy</button>
		&nbsp; <button id='${id}-output-clear'>tear off</button>
		&nbsp; <button><label for='${id}-output-sound'>sound:</label><input type='checkbox' id='${id}-output-sound' checked></button>
	</div>
</div>
`;
}

//---------------------------------------
// Sprite

const EPSILON=1.0/32.0;
const SPR_FRAME_TIME=500;
const SPR_SPEED=32.0/MDH_FPS;
const SPR_INSET=4;
const SPR_HEALTH_BAR_HEIGHT=6;

const SpriteTitleAlign={
	above: "above",
	top: "top",
	middle: "middle",
	bottom: "bottom",
	below: "below",
};

class Sprite {
	constructor(id, pt, size, deftile=null) {
		this.id=id;
		this.pt=pt;
		this.size=size;
		this._pose="stand";
		this.layer=0;
		this._poseTiles={};	// pos := list of tiles for animation
		if (deftile) {
			this._poseTiles[this._pose]=[deftile];
		}
		this.dest=null;
		this.speed=SPR_SPEED;
		this.title=null;
		this.titleColor=null;
		this.titleAlign=SpriteTitleAlign.top;
		this.hp=null;
		this.hpMax=null;
	}

	toString() {
		let s="[Sprite "+this.id+", at "+this.pt+","+this.size+", layer "+this.layer;
		if (this.title) {
			s+=", title "+this.title+","+this.titleColor+","+this.titleAlign;
		}
		s+="]";
		return s;
	}

	bounds() {
		return [this.pt[0], this.pt[1], this.size[0], this.size[1]];
	}

	poseTiles(pose) {
		return this._poseTiles[pose];
	}

	setPoseTiles(pose, tiles) {
		this._poseTiles[pose]=tiles;
	}

	pose() {
		return this._pose;
	}

	setPose(p) {
		this._pose=p;
	}

	/** Force position to animated destination, stop animation.
	 * When turn-based, use this on all sprites before a nextTurn.
	 **/
	snap() {
		if (this.dest) {
			this.pt=this.dest;
			this.dest=null;
		}
	}

	/** Update sprite position.
	 * anim.pt is current location.
	 */
	update(timestamp) {
		if ( ! this.dest) {
			// pass
		} else {
			const delta=[minmax(this.dest[0]-this.pt[0], -this.speed, this.speed),
				minmax(this.dest[1]-this.pt[1], -this.speed, this.speed) ];
			if ( Math.abs(delta[0]) < EPSILON && Math.abs(delta[1]) < EPSILON) {
				this.snap();
			} else {
				this.pt=[this.pt[0] + delta[0], this.pt[1] + delta[1] ];
			}
		}
	}

} // Sprite

//---------------------------------------
// Terminal

class Terminal {
	constructor(id, termsize, charsize, fontname="Fira Code") {
		this.id=id;
		this._termsize=termsize;
		this._charsize=charsize;
		this._fontname=fontname;
		this.size=[termsize[0]*charsize[0], termsize[1]*charsize[1]];

		$(id).innerHTML=termhtml(id, this.size, termsize);

		this._cursor=[0,0];
		this._lastWrapped=false;
		// contents are indexed [y][x],
		// cell is null, or [char,fgcolor,bgcolor]
		this._contents=arrayDim(termsize[1], null);
		for (let y=0; y<termsize[1]; ++y) {
			this._contents[y]=arrayDim(termsize[0], null);
		}
		this._status=null;

		this._imageCache={};		// filename := Image
		this._charTiles={};		// char := [filename,sx,sy,sw,sh]
		this._sprites=[];
		this._pixelFont=null;
		this._pixelMessages=[];	// [text,fg,spt,scale]

		this.cursorOn=true;
		this.fgcolor=TermColor.white;
		this.bgcolor=TermColor.black;
		this.screenBgcolor=TermColor.black;
		this._needsRedraw=true;
		this._lastRedraw=0;
		this.setOutputSound( htmlCookieGet("terminal-output-sound", "1")==="1" );
		this.setKeyboardShown( htmlCookieGet("terminal-key-show", "0")==="1" );

		const elem=$(id+"-canvas");
		this.ctx=elem.getContext("2d");
		this.ctx.mozImageSmoothingEnabled=false;
		this.ctx.webkitImageSmoothingEnabled=false;
		this.ctx.msImageSmoothingEnabled=false;
		this.ctx.imageSmoothingEnabled=false;
		this.ctx.imageSmoothingQuality="low";
		this.ctx.imageRendering="pixelated";
		this.ctx.font=charsize[1]+"px '"+fontname+"'";
		this.ctx.textAlign="left"
		this.ctx.textBaseline="top";

		// NOTE: Safari & Chromium draw text offset differently.
		this.yCharOffset=0;
		if (navigator.userAgent.indexOf("Chrome")>=0) {
			this.yCharOffset=0;
		} else if (navigator.userAgent.indexOf("Safari")>=0) {
			this.yCharOffset= -2;
		}

		const fm=this.ctx.measureText("W");
		console.log("Actual charsize="+fm.width);

		this.keyClear();
		document.addEventListener("focusin", (evt)=>{return this.focusChange(evt, true);}, false);
		document.addEventListener("focusout", (evt)=>{return this.focusChange(evt, false);}, false);
		document.addEventListener("keydown", (evt)=>{return this.keyChange(evt, true);}, false);
		document.addEventListener("keyup", (evt)=>{return this.keyChange(evt, false);}, false);
		document.addEventListener("paste", (evt)=>{return this.clipPaste(evt);}, false);
		document.addEventListener("copy", (evt)=>{return this.clipCopy(evt);}, false);

		$(id+"-output-sound").addEventListener("click", (evt)=>{this.setOutputSound( ! this.outputSound() ); return false;}, false);
		$(id+"-output-clear").addEventListener("click", (evt)=>{this.outputClear(); return false;}, false);
		$(id+"-output-copy").addEventListener("click", (evt)=>{this.outputCopy(evt); return false;}, false);
		$(id+"-key-show").addEventListener("click", (evt)=>{this.setKeyboardShown( ! this.keyboardShown() ); return false;}, false);
		elem.focus();

		this.timer=new Timer( /*update*/(timestamp)=>{ this.redraw(timestamp); },
			/*stop*/()=>{ }
		);
	}

	// properties

	charsize() {
		return this._charsize;
	}

	charTile(c) {
		return this._charTile[c];
	}

	/** Map a character to a tile [filename,sx,sy,sw,sh].
	 * I suggest using characters \u0100 up, so ASCII remains usable.
	 **/
	setCharTile(c, tile) {
		this._charTiles[c]=tile;
	}

	inBounds(pt) {
		return pt[0] >= 0 && pt[0] < this._termsize[0] && pt[1] >= 0 && pt[1] < this._termsize[1];
	}

	keyboardShown() {
		return this._keyboardShown;
	}

	setKeyboardShown(f) {
		this._keyboardShown=f;
		if (f) {
			$(this.id+"-key").classList.remove("hidden");
		} else {
			$(this.id+"-key").classList.add("hidden");
		}
		$(this.id+"-key-show").checked=f;
		htmlCookieSet("terminal-key-show", f?"1":"0");
	}

	lastRedraw() {
		return this._lastRedraw;
	}

	needsRedraw() {
		return this._needsRedraw;
	}

	setNeedsRedraw(f) {
		//DLOG("setNeedsRedraw "+f);
		this._needsRedraw=f;
	}

	outputSound() {
		return this._outputSound;
	}

	setOutputSound(f) {
		this._outputSound=f;
		$(this.id+"-output-sound").checked=f;
		htmlCookieSet("terminal-output-sound", f?"1":"0");
	}

	pixelFont() {
		return this._pixelFont;
	}

	/** Assign filename of pixel font. Must have format "path-black-8.png",
	 * black will be replaced with specific colors, which all must exist.
	 */
	setPixelFont(filename) {
		this._pixelFont=filename;
		for (let k of TERMCOLORKEYS) {
			const f=filename.replace("black", k);
			this.getImage(f);
		}
	}

	sprites() {
		return this._sprites;
	}

	addSprite(spr) {
		this._sprites.push(spr);
	}

	clearSprites() {
		this._sprites=[];
	}

	findSprite(sprid) {
		for (let spr of this._sprites) {
			if (spr.id===sprid) {
				return spr;
			}
		}
		return null;
	}

	removeSprite(spr) {
		arrayRemove(this._sprites, spr);
	}

	termsize() {
		return this._termsize;
	}

	// clipboard

	clipCopy(evt) {
		let text=this.textRange([0,0], [this._termsize[0]-1,this._termsize[1]-1]);
		console.log(text);
		try {
			// FIXME: maybe document.execCommand
			const source=(evt && evt.clipboardData) || window.clipboardData;
			if (source) {
				source.setData("text/plain", text);
				source.setData("text/html", "<pre>"+htmlify(text)+"</pre>");
			}
		} catch (e) {
			console.error("can't copy: "+(evt && ""));
		}
	}

	/** Event responds to paste by inserting all ASCII text, newlines as Enter events.
	 * Ignores non-ASCII, tough.
	 */
	clipPaste(evt) {
		try {
			const source=(evt && evt.clipboardData) || window.clipboardData;
			let text=source.getData("text");
			DLOG("clipPaste "+text);
			for (let c of text) {
				if (c >= ' ' && c <= '~') {
					this.keyQueue.push(c);
				} else if (c==='\r' || c==='\n') {
					this.keyQueue.push("Enter");
				} else {
					console.log("skip char '"+c+"'");
				}
			}
		} catch (e) {
			console.error("can't paste: "+(evt && ""));
		}
	}

	// cursor methods

	/** Move cursor back 1 character (stopping at left edge of screen), erase */
	backspace() {
		const x=Math.max(0, this._cursor[0]-1), y=this._cursor[1];
		this._cursor=[x,y];
		this._contents[y][x]=null;
		this.setNeedsRedraw(true);
	}

	beep() {
		this.synth(250, 440, 0.5, "sawtooth");
	}

	/** Sets current drawing BG color as screen BG color,
	 * clear canvas and contents,
	 * move cursor to 0,0.
	 **/
	cls() {
		this.screenBgcolor=this.bgcolor;
		for (let y=0; y<this._termsize[1]; ++y) {
			this._contents[y].fill(null);
		}
		this.setCursor([0,0]);
		this.clearSprites();
	}

	/** Returns null or [char,fgcolor,bgcolor] */
	contentsAt(pt) {
		return this.inBounds(pt) ? this._contents[pt[1]][pt[0]] : null;
	}

	setContentsAt(pt, con) {
		if (this.inBounds(pt)) {
			this._contents[pt[1]][pt[0]]=con;
			this.setNeedsRedraw(true);
	}}

	cursor() {
		return this._cursor;
	}

	/** Move cursor. Sets needsRedraw. */
	setCursor(pt) {
		if (this.inBounds(pt)) {
			this._cursor=pt;
		} else {
			this._cursor=[0,0];
		}
		this._lastWrapped=false;
		this.setNeedsRedraw(true);
	}

	/** Advance cursor `dx` (default 1) spaces without drawing,
	* wraps around to next line, advances screen if needed.
	*/
	cursorAdvance(dx=1) {
		const t1=Date.now();
		this._lastWrapped=false;
		for (let i=0; i<dx; ++i) {
			let x=this._cursor[0] + 1, y=this._cursor[1];
			if (x >= this._termsize[0]) {
				x=0;
				y+=1;
				this._lastWrapped=true;
				if (y>=this._termsize[1]) {
					this.screenAdvance();
					--y;
			}}
			this._cursor=[x,y];
		}
		this.setNeedsRedraw(true);
		//DLOG("cursorAdvance("+dx+") took "+(Date.now()-t1)+" ms");
	}

	cursorMoveBy(delta) {
		const x=this._cursor[0], y=this._cursor[1];
		this.setCursor([ (x+delta[0]+this._termsize[0]) % this._termsize[0],
						(y+delta[1]+this._termsize[1]) % this._termsize[1] ]);
		this._lastWrapped=false;
	}

	cursorEnd() {
		this.setCursor([this._termsize[0]-1, this.cursor()[1] ]);
	}

	cursorHome() {
		this.setCursor([0, this.cursor()[1] ]); 
	}

	cursorPageUp() {
		let x=this.cursor()[0], y=this.cursor()[1];
		this.setCursor([x, Math.max(0, y-Math.floor(this._termsize[1]/2))]);
	}

	cursorPageDown() {
		let x=this.cursor()[0], y=this.cursor()[1];
		this.setCursor([x, Math.min(this._termsize[1]-1, y+Math.floor(this._termsize[1]/2)) ]);
	}

	/** Pull line back 1 char, blank at end. */
	delforward() {
		const x=this._cursor[0], y=this._cursor[1];
		for (let j=x; j<this._termsize[0]-1; ++j) {
			this._contents[y][j]=this._contents[y][j+1];
		}
		this._contents[y][this._termsize[0]-1]=null;
		this.setNeedsRedraw(true);
	}

	/** Draws a box with line chars. */
	drawLineBox(rect, fg=null, bg=null) {
		if (fg===null) { fg=this.fgcolor; }
		if (bg===null) { bg=this.bgcolor; }
		const x0=Math.floor(rect[0]), x1=Math.floor(rect[0]+rect[2]-1);
		const y0=Math.floor(rect[1]), y1=Math.floor(rect[1]+rect[3]-1);
		if ( !(this.inBounds([x0,y0]) && this.inBounds([x1,y1])) ) {
			console.error("Rect out of bounds "+rect);
			return false;
		}
		this._cursor=[x0,y0]; this.print(TermLineChars[6], fg, bg); // SE
		this._cursor=[x1,y0]; this.print(TermLineChars[12], fg, bg); // SW
		this._cursor=[x0,y1]; this.print(TermLineChars[3], fg, bg); // NE
		this._cursor=[x1,y1]; this.print(TermLineChars[9], fg, bg); // NW
		for (let x=x0+1; x<=x1-1; ++x) {
			this._cursor=[x,y0]; this.print(TermLineChars[10], fg, bg); // EW
			this._cursor=[x,y1]; this.print(TermLineChars[10], fg, bg); // EW
		}
		for (let y=y0+1; y<=y1-1; ++y) {
			this._cursor=[x0,y]; this.print(TermLineChars[5], fg, bg); // NS
			this._cursor=[x1,y]; this.print(TermLineChars[5], fg, bg); // NS
		}
	}

	/** Fills a rectangle with char c. */
	fillCharRect(rect, c, fg=null, bg=null) {
		if (fg===null) { fg=this.fgcolor; }
		if (bg===null) { bg=this.bgcolor; }
		const x0=Math.floor(rect[0]), x1=Math.floor(rect[0]+rect[2]-1);
		const y0=Math.floor(rect[1]), y1=Math.floor(rect[1]+rect[3]-1);
		if ( !(this.inBounds([x0,y0]) && this.inBounds([x1,y1])) ) {
			console.error("Rect out of bounds "+rect);
			return false;
		}
		for (let y=y0; y<=y1; ++y) {
			this.setCursor([x0,y]);
			for (let x=x0; x<=x1; ++x) {
				this.print(c, fg, bg);
			}
		}
	}

	/** Inserts a space at cursor position. Text which scrolls off screen is erased. */
	insert() {
		const x=this._cursor[0], y=this._cursor[1];
		for (let j=this._termsize[0]-1; j>x; --j) {
			this._contents[y][j]=this._contents[y][j-1];
		}
		this._contents[y][x]=null;
		this.setNeedsRedraw(true);
	}

	// TODO: option to suppress advancement
	/** Print text, one char at a time.
	  * \n advances to next line.
	  * fg, bg colors can be set, null = default fgcolor, bgcolor.
	  */
	print(text, fg=null, bg=null) {
		//DLOG(text);
		if (fg===null) { fg=this.fgcolor; }
		if (bg===null) { bg=this.bgcolor; }
		for (let c of text) {
			const x=this._cursor[0], y=this._cursor[1];
			switch (c) {
				case '\x01':
					this.beep();
					break;
				case '\b':
					this.backspace();
					break;
				case '\t':
					this.tab();
					break;
				case '\n':
					if (this._lastWrapped) {
						// newline at right edge = do nothing
					} else {
						this.cursorAdvance(this._termsize[0]-x);
					}
					break;
				default:
					if (c>=" ") {
						const con=[c,fg,bg];
						this.setContentsAt(this._cursor, con);
						this.cursorAdvance(1);
					}
		}}
	}

	/** Scroll screen 1 row up, move cursor up. */
	screenAdvance() {
		const t1=Date.now();
		let firstRow=this._contents[0];
		firstRow.fill(null);
		for (let y=0; y <this._termsize[1]-1; ++y) {
			this._contents[y]=this._contents[y+1];
		}
		this._contents[this._termsize[1]-1]=firstRow;
		const x=this._cursor[0], y=this._cursor[1];
		this.setCursor([x,y-1]);
		//DLOG("screenAdvance() took "+(Date.now()-t1)+" ms");
	}

	/** Display status message on bottom line for short time. */
	status(col, msg) {
		this._status=[Date.now(), col, msg];
	}

	tab() {
		const x=this._cursor[0], y=this._cursor[1];
		this.setCursor([Math.min(this._termsize[0]-1, Math.floor((x+8)/8)*8), y]);
	}

	/** Returns all text in given range of chars. */
	textRange(pt0, pt1) {
		let s=[];
		if ( ! (this.inBounds(pt0) && this.inBounds(pt1)) ) {
			console.error("Invalid textRange "+pt0+", "+pt1);
			return "";
		}
		for (let y=pt0[1]; y<=pt1[1]; ++y) {
			for (let x=pt0[0]; x<=pt1[0]; ++x) {
				let con=this._contents[y][x];
				s.push(con ? con[0] : " ");
			}
			if (y<pt1[1]) {
				s.push("\n");
			}
		}
		return s.join("");
	}

	/** Sets temporary fg,bg colors, executes thunk, restores colors. */
	withColor(fg, bg, thunk) {
		let oldfg=this.fgcolor, oldbg=this.bgcolor;
		if (fg) { this.fgcolor=fg; }
		if (bg) { this.bgcolor=bg; }
		thunk();
		this.fgcolor=oldfg; this.bgcolor=oldbg;
	}

	// event handling

	/** Event handler for focus, state true=in, false=out. */
	focusChange(evt, state) {
		DLOG("focusChange", evt, state);
	}

	/** Event handler for keys, state true=down, false=up. */
	keyChange(evt, state) {
		if (evt.defaultPrevented || evt.metaKey ||
			evt.key==="Shift" || evt.key==="Control" || evt.key==="Alt" || evt.key==="Meta") {
			// ignore
		} else {
			const key=//(evt.shiftKey?"sh-":"")+
				(evt.ctrlKey?"^":"")+ (evt.altKey?"alt-":"")+
				(evt.metaKey?"meta-":"")+ evt.key;
			DLOG("keyChange "+key+" "+state, evt);
			this.handleKey(key, state); // NOTE: evt.timeStamp is imprecise.
			evt.preventDefault();
			return false;
		}
	}

	/** External entry for keys. */
	handleKey(key, state=true, timestamp=null) {
		if ( ! timestamp) {
			timestamp=Date.now();
		}
		if (state) {
			if (key==="meta-c" || key==="^c") {
				return this.clipCopy();
			}
			const t=this.keyState[key];
			if ( ! t) {
				this.keyState[key]=timestamp;
			}

			const debounce=timestamp - (t || 0);
			if (debounce >= KEY_DEBOUNCE_TIME) {
				DLOG("key queued "+key+", debounce="+debounce);
				this.keyQueue.push(key);
				this.keyState[key]=timestamp;
			}
			// if (this.keyQueue.length > 100) {
			// 	//DLOG("key queue flushed");
			// 	this.keyQueue.splice(0, 100);
			// }
		} else {
			this.keyState[key]=null;
		}
	}

	keyClear() {
		this.keyState={};
		this.keyQueue=[];
	}

	// output to fake printer

	/** Clears & hides "-output". */
	outputClear() {
		let out=$(this.id+"-output");
		//out.innerHTML="";
		out.value="";
		$(this.id+"-output-wrapper").classList.add("hidden");
	}

	/** Copy output area to clipboard.
	 * Only does anything on https.
	 */
	outputCopy(evt) {
		let out=$(this.id+"-output");
		//let text=out.innerHTML;
		out.select(); 
		out.setSelectionRange(0, 99999);
		let text=out.value;

		console.log(text);
		if (evt.clipboardData) {
			DLOG("copy to event");
			evt.clipboardData.setData("text/plain", text);
			evt.clipboardData.setData("text/html", "<pre>"+htmlify(text)+"</pre>");
		}
		if (navigator.clipboard) {
			DLOG("copy to navigator");
			navigator.clipboard.writeText(text);
		}
		evt.preventDefault();
		return false;
	}

	/** Writes msg to "-output". */
	output(msg) {
		$(this.id+"-output-wrapper").classList.remove("hidden");
		const out=$(this.id+"-output");
		//out.innerHTML += "<div style='color:"+color+"'>"+msg+"</div>";
		out.value+=msg;
		const atBottom=out.scrollHeight-out.clientHeight<=out.scrollTop+1;
		if ( ! atBottom) {
			out.scrollTop=out.scrollHeight-out.clientHeight;
		}
		if (this._outputSound) {
			this.synth(100, 200, 1.0, "sawtooth", ()=>{ this.synth(100, 400, 1.0, "sawtooth"); } );
		}
	}

	// rendering methods

	/** Returns an Image for a given filename, cached on subsequent calls. */
	getImage(filename) {
		if (this._imageCache[filename]) {
			return this._imageCache[filename];
		}
		DLOG("loading "+filename);
		var img=new Image();
		img.src=filename;
		this._imageCache[filename]=img;
		return img;
	}

	renderChar(pt, con) {
		if (con) {
			const cw=this._charsize[0], ch=this._charsize[1];
			const x=pt[0], y=pt[1], c=con[0], fg=con[1], bg=con[2];
			if (bg !== null) {
				if ( ! TERMCOLORS[bg]) { console.error("Invalid color "+bg); }
				this.ctx.fillStyle=TERMCOLORS[bg];
				this.ctx.fillRect(x*cw, y*ch, cw, ch);
			}
			const ctile=this._charTiles[c];
			if (ctile) {
				this.renderTile(ctile, [x*cw, y*ch, cw, ch]);
				return;
			}
			if ( ! TERMCOLORS[fg]) { console.error("Invalid color "+fg); }
			this.ctx.fillStyle=TERMCOLORS[fg];
			const pix=(c>="\u2580" && c<="\u259f") ? TermPixels.indexOf(c) : 0;
			if (pix) {
				const cw2f=Math.floor(cw/2), ch2f=Math.floor(ch/2);
				const cw2c=Math.ceil(cw/2), ch2c=Math.ceil(ch/2);
				if (pix & 0x01) {	// top left
					this.ctx.fillRect(x*cw, y*ch, cw2c, ch2c);
				}
				if (pix & 0x02) {	// top right
					this.ctx.fillRect(x*cw+cw2c, y*ch, cw2f, ch2c);
				}
				if (pix & 0x04) {	// bottom left
					this.ctx.fillRect(x*cw, y*ch+ch2c, cw2c, ch2f);
				}
				if (pix & 0x08) {	// bottom right
					this.ctx.fillRect(x*cw+cw2c, y*ch+ch2c, cw2f, ch2f);
				}
			} else {
				this.ctx.fillText(c, x*cw, y*ch+this.yCharOffset);
			}
	}}

	/** Clear canvas only. */
	renderClearScreen() {
		this.ctx.fillStyle=TERMCOLORS[this.screenBgcolor];
		this.ctx.fillRect(0, 0, this.size[0], this.size[1]);
	}

	/** Convenience method to call context.lineTo */
	renderDrawLine(pt1,pt2,col=null) {
		if ( ! col) {
			col=this.fgcolor;
		}
		this.ctx.strokeStyle=TERMCOLORS[col];
		this.ctx.beginPath();
		this.ctx.moveTo(pt1[0], pt1[1]);
		this.ctx.lineTo(pt2[0], pt2[1]);
		this.ctx.stroke();
	}

	/** Convenience method to call context.strokeRect */
	renderDrawRect(rect) {
		this.ctx.strokeStyle=TERMCOLORS[this.fgcolor];
		this.ctx.strokeRect(rect[0], rect[1], rect[2], rect[3]);
	}

	/** Convenience method to call context.fillRect */
	renderFillRect(rect) {
		this.ctx.fillStyle=TERMCOLORS[this.fgcolor];
		this.ctx.fillRect(rect[0], rect[1], rect[2], rect[3]);
	}

	renderPixelText(text, srect, fg=null, bg=null) {
		if (bg!==null) {
			this.ctx.fillStyle=TERMCOLORS[bg];
			this.ctx.fillRect(srect[0], srect[1], srect[2], srect[3]);
		}
		if (fg===null) { fg=this.fgcolor; };
		const filename=this._pixelFont.replace("black", TERMCOLORKEYS[fg]);
		this.ctx.fillStyle=TERMCOLORS[fg];
		this.ctx.strokeStyle=TERMCOLORS[fg];
		const scale=toInt(srect[2]/text.length);
		for (let i=0; i<text.length; ++i) {
			let ci=text.codePointAt(i);
			if (ci<0 || ci>255) {
				ci=0;
			}
			const cx=ci%16, cy=Math.floor(ci/16);
			this.renderTile([filename, cx*PIXEL_FONT_SIZE,cy*PIXEL_FONT_SIZE,
								PIXEL_FONT_SIZE,PIXEL_FONT_SIZE],
							[srect[0]+i*scale,srect[1],scale,srect[3]]);
		}
	}

	renderSprites(timestamp) {
		const sprlist=this._sprites.sort( (a,b)=>{
			if (a.layer !== b.layer) {
				return compare(a.layer, b.layer);
			} else {
				return compare(a.id, b.id);
			}
		});

		// Fucking bullshit, doesn't work: ctx.save();
		const ctx=this.ctx;
		const titleFontSize=PIXEL_FONT_SIZE*2;
		const oldFill=ctx.fillStyle, oldStroke=ctx.strokeStyle;

		for (let spr of sprlist) {
			//DLOG("draw "+spr);
			spr.update(timestamp);
			const tiles=spr.poseTiles(spr.pose());
			if (tiles) {
				const frame=Math.floor(timestamp / SPR_FRAME_TIME) % tiles.length;
				this.renderTile(tiles[frame], [spr.pt[0], spr.pt[1], spr.size[0], spr.size[1] ]);
			}
			if (spr.title) {
				let y;
				switch (spr.titleAlign) {
					case SpriteTitleAlign.above: y=spr.pt[1]-titleFontSize; break;
					case SpriteTitleAlign.top: y=spr.pt[1]; break;
					case SpriteTitleAlign.middle: y=spr.pt[1]+toInt((spr.size[1]-titleFontSize)/2); break;
					case SpriteTitleAlign.bottom: y=spr.pt[1]+spr.size[1]-titleFontSize; break;
					case SpriteTitleAlign.below: y=spr.pt[1]+spr.size[1]; break;
				}
				this.renderPixelText(spr.title, [spr.pt[0],y,spr.title.length*titleFontSize,titleFontSize], spr.titleColor, null);
			}
			if (spr.hp) {
				let healthColor=this.healthColor(spr.hp, spr.hpMax);
				if (healthColor>=0) {
					const srect=[spr.pt[0], spr.pt[1]-SPR_HEALTH_BAR_HEIGHT, spr.size[0], SPR_HEALTH_BAR_HEIGHT];
					const healthPx=minmax(spr.hp * spr.size[0] / spr.hpMax, 0, spr.size[0]);
					//DLOG("    health "+healthPx+", color "+healthColor);
					ctx.fillStyle=TERMCOLORS[TermColor.black];
					ctx.fillRect(srect[0], srect[1], srect[2], srect[3]);
					ctx.fillStyle=TERMCOLORS[healthColor];
					ctx.fillRect(srect[0], srect[1], healthPx, srect[3]);
					ctx.strokeStyle=TERMCOLORS[TermColor.graydark];
					ctx.strokeRect(srect[0]-1.0, srect[1]-1.0, srect[2]+2.0, srect[3]+2.0);
				}
			}
		}

		// Fucking bullshit, doesn't work: ctx.restore();
		ctx.fillStyle=oldFill;
		ctx.strokeStyle=oldStroke;
	}

	/** Renders a tile [file,sx,sy,sw,sh] to fill srect, screen coords. */
	renderTile(tile, srect) {
		if ( ! tile) {
			DLOG("ERROR: No tile "+tile+" at "+srect);
			return;
		}
		this.ctx.fillStyle=TERMCOLORS[this.fgcolor];
		const img=this.getImage(tile[0]);
		this.ctx.drawImage(img, tile[1], tile[2], tile[3], tile[4], srect[0], srect[1], srect[2], srect[3]);
	}

	redraw(timestamp) {
		const tw=this._termsize[0], th=this._termsize[1];
		const cw=this._charsize[0], ch=this._charsize[1];
		const ctx=this.ctx;
		this.renderClearScreen();
		for (let y=0; y<th; ++y) {
			for (let x=0; x<tw; ++x) {
				this.renderChar([x,y], this._contents[y][x]);
		}}
		if (this.cursorOn) {
			let ccol=Math.max(1, Math.floor(timestamp/TERMFLICKER) % TERMCOLORS.length); // not black
			ctx.strokeStyle=TERMCOLORS[ccol];
			ctx.strokeRect(this._cursor[0]*cw, this._cursor[1]*ch, cw, ch);
		}
		this._lastRedraw=timestamp;

		if (this._status) {
			if (timestamp - this._status[0] > STATUS_TIME) {
				this._status=null;
			} else {
				let oldc=this.cursor();
				this.setCursor([0,th-1]);
				this.print(this._status[2], TermColor.black, this._status[1]); // inverse
				this.setCursor(oldc);
			}
		}

		if (this._sprites.length > 0) {
			this.renderSprites(timestamp);
		}

		if (this.gridColor) {
			DLOG("drawing grid");
			for (let i=0; i<Math.max(tw,th); ++i) {
				this.renderDrawLine([0,i*ch],[tw*cw,i*ch],this.gridColor);
				this.renderDrawLine([i*cw,0],[i*cw,th*ch],this.gridColor);
			}
		}
		this.setNeedsRedraw(false);
		//DLOG("redraw() took "+(Date.now()-timestamp)+" ms");
	}

	healthColor(hp, hpMax) {
		if (hp===hpMax) {
			// no bar for 100%
			return -1;
		} else if (hp >= hpMax * 0.75) {
			return TermColor.lime;
		} else if (hp >= hpMax * 0.25) {
			return TermColor.yellow;
		} else if (hp >= hpMax * 0.5) {
			return TermColor.orange;
		} else {
			return TermColor.red;
		}
	}

	sizeText(text) {
		const textsize=this.ctx.measureText(text);
		return [Math.ceil(textsize.width), Math.ceil(textsize.height)];
	}

	/** Tests collisions of rectangle dest with everything else,
	 * border is a rect in pixel coords, or null,
	 * being all characters listed in blockingChars,
	 * and all sprite ids except those listed in ignoreSprites (self, your own missiles, etc.),
	 * returns a list of [cause,x,y] (in pixel coords).
	 * "border"
	 * kTiles key
	 * sprite id
	 */
	spriteCollide(dest, border, blockingChars, ignoreSprites) {
		const colls=[];

		const tw=this._termsize[0], th=this._termsize[1];
		const cw=this._charsize[0], ch=this._charsize[1];
		// inset so you can "bounce" into walls a tiny bit.
		const sprcx=Math.floor((dest[0]+SPR_INSET)/cw),
			sprcy=Math.floor((dest[1]+SPR_INSET)/ch),
			sprcx2=Math.ceil((dest[0]+dest[2]-SPR_INSET)/cw),
			sprcy2=Math.ceil((dest[1]+dest[3]-SPR_INSET)/ch);
		//DLOG("dest "+dest+" = "+sprcx+","+sprcy+" to "+sprcx2+","+sprcy2);

		if (border && ! (rectContainsPoint(border, dest) && rectContainsPoint(border, [dest[0]+dest[2]-1,dest[1]+dest[3]-1]) )) {
			colls.push(["border",sprcx,sprcy]);
		}

		for (let y=sprcy; y<sprcy2; ++y) {
			for (let x=sprcx; x<sprcx2; ++x) {
				const pt=[x,y];
				if (this.inBounds(pt)) {
					const con=this.contentsAt(pt), c=con ? con[0] : " ";
					if (blockingChars.indexOf(c)>=0) {
						colls.push([c,x*cw,y*ch]);
					}
				} else {
					colls.push(["border",x*cw,y*ch]);
				}
			}
		}

		for (let other of this._sprites) {
			if (ignoreSprites.indexOf(other.id)<0 && rectIntersectsRect(dest, other.bounds())) {
				colls.push(other.id);
			}
		}

		return colls;
	}

	/** Play synthesized sound
	  * dur: duration in milliseconds.
	  * freq: frequency in hertz.
	  * volume: 0.0-1.0
	  * type: sawtooth, sine, square, triangle, custom.
	  * callback: function played on end.
	  */
	synth(dur=500, freq=440, volume=1.0, type="sine", callback=null) {
		if ( ! this.audio) {
			this.audio=new (window.AudioContext || window.webkitAudioContext || window.audioContext);
		}
		const oscillator=this.audio.createOscillator();
		const gainNode=this.audio.createGain();
		oscillator.connect(gainNode);
		gainNode.connect(this.audio.destination);
		gainNode.gain.value=volume;
		oscillator.frequency.value=freq;
		oscillator.type=type;
		if (callback) { oscillator.onended=callback; }
		oscillator.start(this.audio.currentTime);
		oscillator.stop(this.audio.currentTime + (dur/1000));
	}

} // Terminal

